<?php

namespace Adfox\Subscription\Livewire\Plan;

use App\Models\Plan;
use App\Models\PackageItem;
use App\Settings\PaystackSettings;
use App\Settings\RazorpaySettings;
use Filament\Forms\Components\Radio;
use Livewire\Attributes\Url;
use Livewire\Component;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use App\Settings\StripeSettings;
use App\Settings\PaypalSettings;
use App\Settings\PaymentSettings;
use App\Settings\FlutterwaveSettings;
use App\Settings\OfflinePaymentSettings;
use App\Settings\PaymongoSettings;
use Carbon\Carbon;
use Akaunting\Money\Currency as AkauntingCurrency;
use App\Settings\SubscriptionSettings;

class PlanPayment extends Component implements HasForms
{
    use InteractsWithForms;

    public $currentPayment;
    public $payment_method;
    public $plan;
    public $subtotal = 0;
    public $tax = 0; // Define tax rate if applicable
    public $total = 0;
    public $isDifferentRate = false;
    public $convertedTotal = 0;
    public $defaultCurrency;
    #[Url]
    public $selectedPlan;
    protected $paymentGateways = [
        'stripe' => 'stripe-subscription',
    ];

    /**
     * Initialize the component with the selected plan items and set up payment options.
     *
     * @param array $selectedPlan Array of selected item IDs.
     */
    public function mount($selectedPlan)
    {
        $this->selectedPlan = $selectedPlan;
        $this->initializePaymentOptions();
        $this->updatePaymentData();
    }

    /**
     * Update the payment-related data based on selected plan items.
     */
    protected function updatePaymentData()
    {
        $this->plan = Plan::with('features')->find($this->selectedPlan);
        if(! $this->plan){
            abort(404);
        }
        $this->subtotal = 0;
        $this->subtotal= $this->plan->price;

        // Accessing PaymentSettings
        $paymentSettings = app(PaymentSettings::class);

        if ($paymentSettings->enable_tax) {
            if ($paymentSettings->tax_type === 'percentage') {
                // Calculate tax as a percentage of the subtotal
                $this->tax = ($this->subtotal * $paymentSettings->tax_rate) / 100;
            } else if ($paymentSettings->tax_type === 'fixed') {
                // Apply a fixed tax rate
                $this->tax = $paymentSettings->tax_rate;
            }
        } else {
            // No tax applied
            $this->tax = 0;
        }
        $this->tax = 0;
        // Add tax calculation logic here if necessary
        $this->total = $this->subtotal + $this->tax;


        $this->defaultCurrency =  $paymentSettings->currency;
        $paymentGatewayRate = $this->getPaymentGatewayRate();
        $systemExchangeRate = app(PaymentSettings::class)->exchange_rate;

        $this->isDifferentRate = $paymentGatewayRate != 1.0 && $paymentGatewayRate != $systemExchangeRate;
        $this->convertedTotal = $this->total * $paymentGatewayRate / $systemExchangeRate;
    }

    /**
     * Retrieve the current payment gateway exchange rate.
     *
     * @return float The exchange rate of the selected payment gateway.
     */
    private function getPaymentGatewayRate()
    {
        return match($this->payment_method) {
            'stripe' => app(StripeSettings::class)->exchange_rate,
            'paypal' => app(PaypalSettings::class)->exchange_rate,
            'flutterwave' => app(FlutterwaveSettings::class)->exchange_rate,
            'offline' =>  app(OfflinePaymentSettings::class)->exchange_rate,
            'paymongo' => app(PaymongoSettings::class)->exchange_rate,
            'razorpay' => app(RazorpaySettings::class)->exchange_rate,
            'paystack' => app(PaystackSettings::class)->exchange_rate,
            default => 1.0
        };
    }

    public function getExchangeCurrencySymbol()
    {
        return match($this->payment_method) {
            'stripe' =>(new AkauntingCurrency(app(StripeSettings::class)->currency))->getSymbol() ,
            'paypal' =>(new AkauntingCurrency(app(PaypalSettings::class)->currency))->getSymbol() ,
            'flutterwave' =>(new AkauntingCurrency(app(FlutterwaveSettings::class)->currency))->getSymbol() ,
            'offline' =>(new AkauntingCurrency(app(OfflinePaymentSettings::class)->currency))->getSymbol() ,
            'paymongo' =>(new AkauntingCurrency(app(PaymongoSettings::class)->currency))->getSymbol() ,
            'razorpay' =>(new AkauntingCurrency(app(RazorpaySettings::class)->currency))->getSymbol() ,
            'paystack' =>(new AkauntingCurrency(app(PaystackSettings::class)->currency))->getSymbol() ,
            default => '$'
        };
    }

    /**
     * Initialize available payment options based on system settings.
     *
     * @return array Array of available payment options.
     */
    protected function initializePaymentOptions()
    {
        $paymentOptions = [];

        if (app(StripeSettings::class)->status && app(SubscriptionSettings::class)->status) {
            $paymentOptions['stripe'] = app(StripeSettings::class)->name;
        }


        // Set default payment method if only one option is enabled
        if (count($paymentOptions) === 1) {
            $defaultMethod = array_key_first($paymentOptions);
            $this->payment_method = $defaultMethod;
            $this->currentPayment = $this->paymentGateways[$defaultMethod];
        }

        return $paymentOptions;
    }

     /**
     * Method to get payment data in the specified format.
     *
     * @return array
     */
    public function getPaymentDataProperty()
    {
        return [
            'user_id' => auth()->id(),
            'plan_id' => $this->selectedPlan,
            'price_id' => $this->plan->price_id,
            'subtotal' => $this->subtotal,
            'tax' => $this->tax,
            'total' => $this->total
        ];
    }

    /**
     * Define the form schema for payment method selection.
     *
     * @param Form $form The filament form instance.
     * @return Form The configured form schema.
     */
    public function form(Form $form): Form
    {
        $paymentOptions = $this->initializePaymentOptions();

        return $form
            ->schema([
                Radio::make('payment_method')
                ->hiddenLabel()
                ->live()
                ->options($paymentOptions)
                ->afterStateUpdated(function ($state) {
                    $this->currentPayment = $this->paymentGateways[$state] ?? null;
                    $this->updatePaymentData();
                }),
            ]);
    }

    /**
     * Render the component view.
     *
     * @return \Illuminate\View\View The view to render.
     */
    public function render()
    {
        return view('subscription::plan.plan-payment');
    }
}
