<?php

namespace Adfox\Subscription\Filament\Pages;

use App\Settings\SubscriptionSettings;
use App\Settings\RazorpaySettings;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Pages\SettingsPage;
use App\Models\SettingsProperty;
use App\Settings\PackageSettings;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Set;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\HtmlString;

class SubscriptionSetup extends SettingsPage
{
    protected static ?string $title = 'Subscription Settings';

    protected static ?string $navigationGroup = 'Subscription Management';

    protected static ?string $navigationLabel = 'Subscription Settings';

    protected static ?string $slug = 'manage-subscription-settings';

    protected static string $settings = SubscriptionSettings::class;

    protected static ?int $navigationSort = 3; // Adjust the sort order as needed

    protected function mutateFormDataBeforeSave(array $data): array
    {
        $previousData = app(SubscriptionSettings::class);
        $filtered = [];

        foreach ($data as $key => $item) {
            if (property_exists($previousData, $key)) {
                $propertyType = gettype($previousData->$key);

                if (is_null($item) && $propertyType === 'string') {
                    $filtered[$key] = '';
                    continue;
                }
            }
            $filtered[$key] = $item;
        }
        if(!$filtered['status']){
            $filtered['combine_subscriptions_and_orders']=false;
        }
        return $filtered;
    }

    public function afterSave()
    {
        //Disable package if subscription enabled
        $packageSettings = app(PackageSettings::class);
        if (isset($this->data['status']) && $this->data['status']) {
            $packageSettings->status = false;
            $packageSettings->save();
        }
    }

    public function form(Form $form): Form
    {
        $currenciesConfig = config('money.currencies');
        $currencyCodes = array_keys($currenciesConfig);
        $isDemo = Config::get('app.demo');

        return $form
            ->schema([
                Toggle::make('status')
                    ->label('Enable Subscription')
                    ->live()
                    ->afterStateUpdated(fn($state,Set $set)=> !$state?$set('combine_subscriptions_and_orders',false):'')
                    ->helperText('Toggle to enable or disable Subscription.'),
                Toggle::make('combine_subscriptions_and_orders')
                    ->helperText('If enabled, the My-Subscription page will render within the My-Order page. If disabled, it will have a separate URL.'),
                TextInput::make('free_ad_limit')
                    ->numeric()
                    ->minValue(0)
                    ->label('Free Ad Limit')
                    ->placeholder('Enter the number of free ads allowed')
                    ->required(),
                $isDemo ?
                Placeholder::make('stripe_webhook_secret')
                    ->label('Stripe Webhook Secret')
                    ->content('*****')
                    ->hint('Hidden due to demo mode.') :
                TextInput::make('stripe_webhook_secret')
                    ->label('Stripe Webhook Secret')
                    ->required()
                    ->helperText(new HtmlString("This secret key is used by Stripe to securely notify your application about events. Keep it confidential!. <a style='text-decoration:underline' target='blank' href='https://docs.stripe.com/webhooks'><b>Click here</b></a> to learn more about Stripe webhooks."))
                    ->hint(str('[Stripe Webhook](https://dashboard.stripe.com/account/webhooks)')->inlineMarkdown()->toHtmlString())
                    ->hintColor('warning')
                    ->hintIcon('heroicon-m-question-mark-circle', tooltip: 'Click it!'),
                Section::make('Instructions to Add Stripe Webhook Endpoint and Configure Events')
                    ->collapsed()
                    ->schema([
                        Placeholder::make('')
                            ->columnSpanFull()
                            ->content(self::getWebhookInstruction())
                    ])
            ])
            ->columns(2);
    }

    protected static function getWebhookInstruction()
    {
        return new HtmlString('<div class="prose">
    <h3> Create a Webhook Endpoint in Stripe</h3>
    <ol>
        <li>Login to Stripe Dashboard:
            <ul>
                <li>Go to <a style="text-decoration:underline" target="blank" href="https://dashboard.stripe.com">Stripe Dashboard</a>.</li>
                <li>Log in with your Stripe account credentials.</li>
            </ul>
        </li>
        <li>Navigate to Webhooks:
            <ul>
                <li>In the left-hand menu, click on "Developers".</li>
                <li>Under "Developers", click on "Webhooks".</li>
            </ul>
        </li>
        <li>Add an Endpoint:
            <ul>
                <li>Click the "Add endpoint" button.</li>
            </ul>
        </li>
        <li>Set the Endpoint URL:
            <ul>
                <li>In the "Endpoint URL" field, enter your site URL followed by <code>/stripe/webhook</code>. For example, if your site URL is <code>' . url('/') . '</code>, the endpoint URL should be <code>' . url('/') . '/stripe/webhook</code>.</li>
            </ul>
        </li>
        <li>Select Events to Listen To:
            <ul>
                <li>Click the "Select events" button.</li>
                <li>Choose the following events:
                    <ul>
                        <li><code>customer.subscription.created</code></li>
                        <li><code>customer.subscription.updated</code></li>
                        <li><code>customer.subscription.deleted</code></li>
                        <li><code>invoice.created</code></li>
                        <li><code>invoice.payment_succeeded</code></li>
                        <li><code>invoice.paid</code></li>
                        <li><code>invoice.payment_failed</code></li>
                    </ul>
                </li>
            </ul>
        </li>
        <li>Save Endpoint:
            <ul>
                <li>After selecting the events, click the "Add endpoint" button to save the configuration.</li>
            </ul>
        </li>
    </ol>
    </div>');
    }
}
