<?php

namespace Adfox\Razorpay\Livewire;

use App\Models\WebhookPackage;
use App\Traits\HandlesPackageApplication;
use App\Traits\HandlesPromotionApplication;
use Livewire\Attributes\On;
use Livewire\Component;
use App\Models\WebhookUpgrade;
use App\Settings\PaymentSettings;
use App\Settings\RazorpaySettings;
use App\Settings\GeneralSettings;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Session;
use Razorpay\Api\Api;
use Razorpay\Api\Errors\SignatureVerificationError;

class RazorpayPayment extends Component
{
    use HandlesPackageApplication;
    use HandlesPromotionApplication;
    public $id;

    public $type;

    public $total;

    public $data;

    public $tax;

    public $subtotal;

    public $successUrl;

    public $payment_gateway_params = [];

    /**
     * Mount the component and set the properties if an ad ID is provided.
     */
    public function mount($id)
    {
        $this->id = $id;
        $this->processPayment();
        $this->successUrl = url('callback/razorpay');
    }

    public function getPaymentSettingsProperty()
    {
        return app(PaymentSettings::class);
    }

    public function getGeneralSettingsProperty()
    {
        return app(GeneralSettings::class);
    }

    public function getRazorpaySettingsProperty()
    {
        return app(RazorpaySettings::class);
    }

    public function processPayment()
    {

        $request = new Api($this->getRazorpaySettingsProperty()->key_id,$this->getRazorpaySettingsProperty()->key_secret);

        $currency=$this->getRazorpaySettingsProperty()->currency;
        // Create order
        $order = $request->order->create([
            'amount' =>  (int)$this->total *100,
            'currency' =>  $currency,
        ]);

        // Set Razorpay payment parameters
        $this->payment_gateway_params['order_id'] = $order->id;

        $this->handleWebhookUpgrade($order->id);
    }


    protected function handleWebhookUpgrade($order_id)
    {
        try {
            if($this->type == 'PKG') {
                WebhookPackage::create([
                    'data' => json_encode($this->data),
                    'payment_id' => $order_id,
                    'payment_method' => 'razorpay',
                    'status' => 'pending'
                ]);
            } else {
                WebhookUpgrade::create([
                    'data' => json_encode($this->data),
                    'payment_id' => $order_id,
                    'payment_method' => 'razorpay',
                    'status' => 'pending'
                ]);
            }
        } catch (\Throwable $th) {
            // Handle any exceptions
        }
    }

    #[On('razorpay-handler')]
    public function verifyPayment($response){
        $success = true;
        if (isset($response['razorpay_payment_id'])) {
            $api = new Api($this->getRazorpaySettingsProperty()->key_id,$this->getRazorpaySettingsProperty()->key_secret);
            try {
                //verify the payment
                $api->utility->verifyPaymentSignature($response);
            } catch (SignatureVerificationError $e) {
                $success = false;
                return [
                    'success' => false,
                    'message' => __('messages.t_toast_something_went_wrong')
                ];
            }
        }

        if ($success === true) {
            // Get order id
            $order_id = $response['razorpay_order_id'];
            // Check If Package Management Callback
            if($this->type == 'PKG') {
                $orderData = WebhookPackage::where('payment_id', $order_id)
                ->where('payment_method', 'razorpay')
                ->where('status', 'pending')
                ->firstOrFail();

                $routeParameters = $this->applyPackages($orderData);
                return redirect()->route('package-success', $routeParameters);

            } else {

                $orderData = WebhookUpgrade::where('payment_id', $order_id)
                                        ->where('payment_method', 'razorpay')
                                        ->where('status', 'pending')
                                        ->firstOrFail();

                $routeParameters = $this->applyPromotionsWithRedirect($orderData);

                return redirect()->route('success-upgrade', $routeParameters);
            }
        }

    }

    /**
     * Render the component view.
     *
     * @return \Illuminate\View\View The view to render.
     */
    public function render()
    {
        return view('razorpay::razorpay-payment');
    }
}
