<?php

namespace Adfox\Packages\Livewire;

use App\Models\Ad;
use App\Models\AdPromotion;
use App\Models\OrderPackageItem;
use App\Models\Package;
use App\Models\PackageItem;
use App\Models\UsedPackageItem;
use Carbon\Carbon;
use Filament\Notifications\Notification;
use Livewire\Component;
use Illuminate\Support\Collection;
use Livewire\Attributes\On;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Forms\Components\TextInput;

class PackageDetails extends Component implements HasForms
{
    use InteractsWithForms;

    public $packages;
    public $website_url;
    public $website_label;
    public $pkg_type;
    public $ad_id;
    public $groupedPromotions;
    public $activeOrderPackageItems;
    public $selectedItems = [];

    /**
     * Mount the component and load the necessary data based on package type.
     *
     * @param string $pkg_type The type of the package.
     */
    public function mount()
    {
        $this->packages = Package::with(['packageItems', 'packagePromotions.promotion'])
            ->where('is_enabled', true)
            ->get();

        if ($this->pkg_type === 'single') {
            $this->groupedPromotions = $this->groupSinglePromotions();
        } elseif ($this->pkg_type === 'apply') {
            $this->activeOrderPackageItems = $this->loadActiveOrderPackageItems();
        }

        if ($this->ad_id) {
            $this->loadAdDetails($this->ad_id);
        }
    }

       /**
     * Load and set ad details if an ID is provided.
     *
     * @param int $id The ID of the ad to load
     */
    private function loadAdDetails($id)
    {
        $ad = Ad::find($id);
        if ($ad) {
          $this->website_url = $ad->website_url;
          $this->website_label = $ad->website_label;
        }
    }

    /**
     * Group promotions with single items.
     *
     * @return Collection Grouped promotions with single items.
     */
    protected function groupSinglePromotions(): Collection
    {
        // Eager load the packagePromotions and their related packageItems
        $this->packages->load(['packagePromotions.packageItems']);

        $promotionsWithSingleItems = collect();

        // Iterate through each package and its promotions
        foreach ($this->packages as $package) {
            foreach ($package->packagePromotions as $packagePromotion) {
                // Filter the packageItems for this promotion where quantity is 1
                $singleItems = $packagePromotion->packageItems->where('quantity', 1);
                foreach ($singleItems as $item) {
                    // Add the promotion and its single items to the collection
                    $promotionsWithSingleItems->push([
                        'promotion' => $packagePromotion->promotion,
                        'item' => $item
                    ]);
                }
            }
        }

        // Group the results by promotion ID
        return $promotionsWithSingleItems->groupBy('promotion.id')->map(function ($groupedItems) {
            // Return the promotion and its items
            return [
                'promotion' => $groupedItems->first()['promotion'],
                'items' => $groupedItems->pluck('item')
            ];
        });
    }

    /**
     * Load active order package items that belong to the authenticated user.
     *
     * @return Collection Active order package items.
     */
    protected function loadActiveOrderPackageItems(): Collection
    {
        return OrderPackageItem::whereHas('orderPackage', function ($query) {
            $query->where('user_id', auth()->id());
        })
        ->whereDate('activation_date', '<=', now())
        ->whereDate('expiry_date', '>=', now())
        ->where('available', '>', 0)
        ->where('type', 'promotion')
        ->get();
    }

    /**
     * Toggle the selection of package items.
     *
     * @param int $itemId The ID of the package item to toggle.
     */
    public function toggleSelection($itemId)
    {

        if ($this->pkg_type === 'single' || $this->pkg_type === 'apply') {
            // For 'single' type, only allow one item to be selected at a time.
            $this->selectedItems = [$itemId];
        } else {
            // For other types, allow multiple selections.
            if (in_array($itemId, $this->selectedItems)) {
                // If the item is already selected, deselect it.
                $this->selectedItems = array_diff($this->selectedItems, [$itemId]);
            } else {
                // If the item is not selected, add it to the selection.
                $this->selectedItems[] = $itemId;
            }
        }

        // Reset array keys to maintain an indexed array
        $this->selectedItems = array_values($this->selectedItems);
        $this->dispatch('selected-package-items', $this->selectedItems);
    }


     /**
     * Move to the next step in the process.
     */
    #[On('proceed-next')]
    public function proceedNext() {


        if($this->pkg_type === 'single') {
            $packageItem = PackageItem::find($this->selectedItems[0]);

            if($this->pkg_type === 'single' && empty($this->website_url) && $packageItem->promotion->promotion->id == 4) {
                $this->addError('website_url', __('messages.t_enter_website_url'));
                return;
            }

            // Check for existing promotion
            $existingPromotion = AdPromotion::where('ad_id', $this->ad_id)
            ->where('promotion_id', $packageItem->promotion->promotion->id)
            ->first();

            // If existing promotion found and it's still active (end date is not in the past)
            if ($existingPromotion && !$existingPromotion->end_date->isPast()) {
                Notification::make()
                ->title(__('messages.t_promotion_already_active'))
                ->danger()
                ->send();

                // Redirect or perform some other action here
                return back();
            }
        }
        $this->dispatch('next-step');
    }

    #[On('apply-package')]
    public function applyPackage() {
       // Find orderPackageItem
       $orderPackageItem = OrderPackageItem::find($this->selectedItems[0]);
       $packageItem = PackageItem::find($orderPackageItem->package_item_id);

       // Its only for website URL
       if (empty($this->website_url) && $packageItem->promotion->promotion_id == 4) {
            $this->addError('website_url', __('messages.t_enter_website_url'));
            return;
        }

       // Check for existing promotion
       $existingPromotion = AdPromotion::where('ad_id', $this->ad_id)
                           ->where('promotion_id', $packageItem->promotion->promotion->id)
                           ->first();

       // If existing promotion found and it's still active (end date is not in the past)
       if ($existingPromotion && !$existingPromotion->end_date->isPast()) {
           Notification::make()
               ->title(__('messages.t_promotion_already_active'))
               ->danger()
               ->send();

           // Redirect or perform some other action here
           return back();
       }


       // If no active existing promotion, create new promotion
       AdPromotion::create([
           'ad_id' => $this->ad_id,
           'promotion_id' => $packageItem->promotion->promotion->id,
           'start_date' => Carbon::now(),
           'end_date' => $orderPackageItem->expiry_date,
           'order_package_item_id' => $orderPackageItem->id,
           'price' => $orderPackageItem->price
       ]);

       $orderPackageItem->decrement('available');
       $orderPackageItem->increment('used');

       UsedPackageItem::create([
            'ad_id' =>  $this->ad_id,
            'order_package_item_id' =>  $orderPackageItem->id,
       ]);

       $routeParameters = ['actionType' => 'apply-package', 'ad_id' => $this->ad_id];
       return redirect()->route('package-success', $routeParameters);
    }


     /**
     * Define the form for the website URL input.
     *
     * @param Form $form
     * @return Form
     */
    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('website_url')
                    ->label(__('messages.t_website_url'))
                    ->live(onBlur: true)
                    ->placeholder(__('messages.t_enter_your_business_website'))
                    ->minLength(10)
                    ->required(),
                TextInput::make('website_label')
                    ->label(__('messages.t_website_label'))
                    ->live(onBlur: true)
                    ->placeholder(__('messages.t_enter_website_label'))
                    ->minLength(2)
                    ->required(),
            ]);
    }

    public function updatedWebsiteUrl($value) {
        $ad = Ad::find($this->ad_id);
        if ($ad) {
            $ad->update(['website_url' => $value]);
        }
    }

    public function updatedWebsiteLabel($value) {
        $ad = Ad::find($this->ad_id);
        if ($ad) {
            $ad->update(['website_label' => $value]);
        }
    }

    /**
     * Render the component view.
     *
     * @return \Illuminate\View\View The view to render.
     */
    public function render()
    {
        return view('packages::package-details');
    }
}
