<?php

namespace Adfox\Blog\Livewire;

use App\Models\BlogPost;
use App\Models\PostCategory;
use App\Settings\BlogSettings;
use App\Settings\GeneralSettings;
use App\Settings\SEOSettings;
use Livewire\Component;
use Livewire\WithPagination;
use Artesaos\SEOTools\Traits\SEOTools as SEOToolsTrait;
use Livewire\Attributes\Url;

class BlogComponent extends Component
{

    use WithPagination, SEOToolsTrait;

    #[Url(as: 'ref', except: '/')]
    public $referrer = '/';

    #[Url(as: 'query')]
    public $search;
    public $categories;

    #[Url(as: 'category')]
    public $mainCategory;
    /**
     * Init component
     *
     * @return void
     */
    public function mount()
    {
        // Check if blog system enabled
        if (!app(BlogSettings::class)->enable_blog) {
            return redirect('/');
        }
        $this->loadCategories();
        $this->setSEOData();
    }

    public function loadCategories()
    {
        $this->categories = PostCategory::all()->sortBy('order');
    }

    public function selectMainCategory($slug)
    {
        $this->mainCategory = $slug == $this->mainCategory ? null : $slug;
        $this->getArticlesProperty();
    }

    /**
     * Render component
     *
     * @return Illuminate\View\View
     */
    public function render()
    {

        return view('blog::blog', [
            'articles' => $this->articles
        ]);
    }


    /**
     * Set SEO data
     */
    protected function setSeoData()
    {
        $generalSettings = app(GeneralSettings::class);
        $seoSettings = app(SEOSettings::class);

        $siteName = $generalSettings->site_name ?? app_name();
        $separator = $generalSettings->separator ?? '-';

        $title = __('messages.t_blogs') . " $separator " . $siteName;
        $description = $seoSettings->meta_description ?? app_name();
        $ogImage = getSettingMediaUrl('seo.ogimage', 'seo', asset('images/ogimage.jpg'));
        $this->seo()->setTitle($title);
        $this->seo()->setDescription($description);
        $this->seo()->setCanonical(url()->current());
        $this->seo()->opengraph()->setTitle($title);
        $this->seo()->opengraph()->setDescription($description);
        $this->seo()->opengraph()->setUrl(url()->current());
        $this->seo()->opengraph()->setType('website');
        $this->seo()->opengraph()->addImage($ogImage);
        $this->seo()->twitter()->setImage($ogImage);
        $this->seo()->twitter()->setUrl(url()->current());
        $this->seo()->twitter()->setSite("@" . $seoSettings->twitter_username);
        $this->seo()->twitter()->addValue('card', 'summary_large_image');
        $this->seo()->metatags()->addMeta('fb:page_id', $seoSettings->facebook_page_id, 'property');
        $this->seo()->metatags()->addMeta('fb:app_id', $seoSettings->facebook_app_id, 'property');
        $this->seo()->metatags()->addMeta('robots', 'index, follow, max-image-preview:large, max-snippet:-1, max-video-preview:-1', 'name');
        $this->seo()->jsonLd()->setTitle($title);
        $this->seo()->jsonLd()->setDescription($description);
        $this->seo()->jsonLd()->setUrl(url()->current());
        $this->seo()->jsonLd()->setType('WebSite');
    }

    /**
     * Get posts
     *
     * @return object
     */
    public function getArticlesProperty()
    {
        $filters = [
            'search' => $this->search,
            'main_category' => $this->mainCategory
        ];
        return BlogPost::query()->filter($filters)->withCount('comments')->latest()->paginate(20);
        ;
    }

    public function performSearch()
    {
        $this->getArticlesProperty();
    }
}
