<?php

namespace App\Livewire\Ad\PostAd;

use App\Forms\Components\GroupedFields;
use App\Foundation\AdBase\BaseAdDetail;
use App\Models\Ad;
use App\Models\PriceType;
use App\Models\AdCondition;
use Filament\Forms\Components\Hidden;
use Livewire\Component;
use Filament\Forms\Components\MarkdownEditor;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Components\ToggleButtons;
use Ysfkaya\FilamentPhoneInput\Forms\PhoneInput;

class AdDetailNew extends BaseAdDetail
{
    protected function fillAdDetails()
    {
        $this->title = $this->ad->title;
        $this->description = $this->ad->description;
        $this->price = $this->ad->price;
        $this->type = $this->ad->type;
        $this->for_sale_by = $this->ad->for_sale_by;
        $this->tags = $this->ad->tags ? $this->ad->tags : [];
        $this->price_type_id = $this->ad->price_type_id;
        $this->display_phone = $this->ad->display_phone;
        $this->phone_number = $this->ad->phone_number;
        $this->condition_id = $this->ad->condition_id;
        $this->price_suffix = $this->ad->price_suffix;
        $this->offer_price = $this->ad->offer_price;

        if ($this->ad->category_id) {
            $this->category_id = $this->ad->category_id;
            if ($this->ad->category) {
                $this->parent_category = $this->ad->category->parent_id;
                $this->showMainCategories = false;
                $this->disable_condition = $this->ad->adType?->disable_condition;
                $this->disable_price_type = $this->ad->adType?->disable_price_type;
            }
        }
    }

    public function titleInput(): Form
    {
        return $this->makeForm()
            ->schema([
                TextInput::make('title')
                    ->label(__('messages.t_title'))
                    ->live(onBlur: true)
                    ->placeholder(__('messages.t_what_are_you_selling'))
                    ->minLength(10)
                    ->maxLength(70)
                    ->required(),
            ]);
    }

    public function detailForm(): Form
    {
        return $this->makeForm()
            ->schema([
                Hidden::make('category_id')->live(),
                ToggleButtons::make('for_sale_by')
                    ->label(__('messages.t_for_sale_by'))
                    ->live()
                    ->grouped()
                    ->options([
                        'owner' => __('messages.t_owner_for_sale'),
                        'business' => __('messages.t_business_for_sale'),
                    ]),
                MarkdownEditor::make('description')
                    ->disableToolbarButtons(['attachFiles'])
                    ->label(__('messages.t_description'))
                    ->live(onBlur: true)
                    ->minLength(20)
                    ->required(),
                ToggleButtons::make('condition_id')
                    ->hidden(fn(Component $livewire) => $livewire->disable_condition)
                    ->label(__('messages.t_condition'))
                    ->live()
                    ->options(AdCondition::all()->pluck('name', 'id'))
                    ->inline(),
                Select::make('price_type_id')
                    ->hidden(fn(Component $livewire) => $livewire->disable_price_type)
                    ->selectablePlaceholder(false)
                    ->label(__('messages.t_price'))
                    ->live()
                    ->required()
                    ->native(false)
                    ->options(function () {
                        $category = $this->getCurrentCategory();
                        if ($category && $category->customize_price_type) {
                            return PriceType::whereIn('id', $category->price_types ?? [])->pluck('name', 'id') ?? [];
                        } else {
                            return PriceType::pluck('name', 'id');
                        }
                    }),
                Select::make('price_suffix')
                    ->default('days')
                    ->label(__('messages.t_price'))
                    // ->hidden(!is_vehicle_rental_active())
                    ->required()
                    ->default('day')
                    ->options(['day' => __('messages.t_per_day')]),
                TextInput::make('price')
                    ->hiddenLabel()
                    ->helperText(__('messages.t_set_fair_price'))
                    ->required()
                    ->validationAttribute(__('messages.t_price'))
                    ->numeric()
                    ->minValue(1)
                    ->placeholder(__('messages.t_price_your_ad'))
                    ->prefix(config('app.currency_symbol'))
                    ->live(onBlur: true)
                    ->hidden(function (Get $get, Component $livewire) {
                        $category = $this->getCurrentCategory();
                        if (!$category) {
                            return false;
                        }
                        if ($livewire->disable_price_type || $get('price_type_id') != 1 || $category->has_price_suffix) {
                            return true;
                        }
                    }),
                TextInput::make('offer_price')
                    ->label(__('messages.t_offer_price'))
                    ->helperText(__('messages.t_set_offer_price'))
                    ->validationAttribute(__('messages.t_offer_price'))
                    ->numeric()
                    ->minValue(1)
                    ->lt('price')
                    ->placeholder(__('messages.t_enter_offer_price'))
                    ->prefix(config('app.currency_symbol'))
                    ->live(onBlur: true)
                    ->hidden(function (Get $get, Component $livewire) {
                        $category = $this->getCurrentCategory();
                        if (!$category) {
                            return false;
                        }
                        if ($livewire->disable_price_type || $get('price_type_id') != 1 || (!$category->enable_offer)) {
                            return true;
                        }
                    }),
                GroupedFields::make([
                    TextInput::make('price')
                        ->helperText(__('messages.t_set_fair_price'))
                        ->required()
                        ->validationAttribute(__('messages.t_price'))
                        ->numeric()
                        ->minValue(1)
                        ->placeholder(__('messages.t_price_your_ad'))
                        ->prefix(config('app.currency_symbol'))
                        ->live(onBlur: true)
                        ->hiddenLabel(),
                    Select::make('price_suffix')
                        ->required()
                        ->options(function () {
                            $category = $this->getCurrentCategory();
                            return $category ? array_combine($category->suffix_field_options, $category->suffix_field_options) : [];
                        })
                ])
                    ->helperText(__('messages.t_set_fair_price'))
                    ->hiddenLabel()
                    ->hidden(function (Get $get, Component $livewire) {
                        $category = $this->getCurrentCategory();
                        if (!$category) {
                            return true;
                        }
                        if ($livewire->disable_price_type || $get('price_type_id') != 1 || (!$category->has_price_suffix)) {
                            return true;
                        }
                    }),
                ToggleButtons::make('display_phone')
                    // ->hidden(fn() => is_vehicle_rental_active())
                    ->label(__('messages.t_display_phone_number'))
                    ->live()
                    ->boolean()
                    ->grouped(),
                PhoneInput::make('phone_number')
                    ->placeholder(__('messages.t_enter_phone_number'))
                    ->helperText(__('messages.t_phone_number_helper'))
                    ->required()
                    ->hidden(fn(Get $get): bool => !$get('display_phone'))
                    ->hiddenLabel(),
                TagsInput::make('tags')
                    // ->hidden(fn() => is_vehicle_rental_active())
                    ->label(__('messages.t_tags'))
                    ->helperText(__('messages.t_set_tags'))
                    ->live(onBlur: true),
            ]);
    }

    public function businessSpecificForm(): Form
    {
        // Regular ads don't have business-specific fields
        return $this->makeForm()->schema([]);
    }
    public function updated($name, $value)
    {
        $this->checkRequiredFieldsFilled();
        $userId = auth()->id();
        $this->validateOnly($name);
        if (!$this->id) {
            if (!$userId) {
                abort(403, 'Unauthorized action.');
            }
            $ad = Ad::create([$name => $value, 'user_id' => $userId]);
            $this->id = $ad->id;
            $this->updateAdSlug($ad, $value);
            $this->loadAdDetails($this->id);
        } else {
            $ad = Ad::find($this->id);

            if (!$ad || $ad->user_id != $userId) {
                abort(403, 'Unauthorized action.');
            }
            // Update the "tags" property if it's an array element update
            if (str_starts_with($name, 'tags.')) {
                $index = explode('.', $name)[1];
                $tags = $this->tags;
                $tags[$index] = $value;
                $this->tags = $tags;
                $ad->update(['tags' => json_encode($tags)]);
            } else {
                // Update other fields
                $adData = [$name => $value];
                $ad->update($adData);
                if ($name == 'title') {
                    // If the title is updated, update the slug too
                    $this->updateAdSlug($ad->fresh(), $value);
                }
                if ($name == 'category_id') {
                    $this->disable_condition = $ad->adType?->disable_condition;
                    $this->disable_price_type = $ad->adType?->disable_price_type;

                    // If the sub category is updated, update the location if it has default location
                    $this->updateLocation($ad, $value);
                }
            }
        }
        $this->dispatch('ad-created', id: $ad->id);
    }

    public function save()
    {
        $this->validate();

        $adData = $this->only([
            'title',
            'category_id',
            'description',
            'price',
            'price_type_id',
            'condition_id',
            'display_phone',
            'phone_number',
            'for_sale_by',
            'tags',
            'price_suffix',
            'offer_price'
        ]);

        if ($this->id) {
            $ad = Ad::find($this->id);
            $ad->update($adData);
        } else {
            $ad = Ad::create(array_merge($adData, ['user_id' => auth()->id()]));
        }

        $this->updateAdSlug($ad, $this->title);

        $this->dispatch('ad-saved', id: $ad->id);
        return $this->redirect('/ads');
    }
}
