<?php

namespace App\Filament\Resources;

use App\Filament\Clusters\AdManagement;
use App\Filament\Resources\ModificationResource\Pages;
use App\Filament\Resources\ModificationResource\RelationManagers;
use App\Models\Media;
use Approval\Models\Modification;
use Filament\Forms;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Form;
use Filament\Infolists\Components\Grid;
use Filament\Infolists\Components\ImageEntry;
use Filament\Infolists\Components\TextEntry;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;

class ModificationResource extends Resource
{
    protected static ?string $model = Modification::class;

    protected static ?string $navigationIcon = 'heroicon-o-pencil-square';

    protected static ?string $cluster = AdManagement::class;

    public static function getModelLabel(): string
    {
        return __('messages.t_ap_modification');
    }

    public static function canAccess(): bool
    {
        return userHasPermission('view_modification');
    }

    public static function canCreate(): bool
    {
        return userHasPermission('create_modification');
    }

    public static function canEdit($record): bool
    {
        return userHasPermission('update_modification');
    }

    public static function canDelete($record): bool
    {
        return userHasPermission('delete_modification');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(fn(Builder $query) => $query->activeOnly())
            ->columns([
                TextColumn::make('ad')
                    ->label(__('messages.t_ap_ad'))
                    ->limit(35)
                    ->default(function ($record) {
                        if ($record->modifiable_type == 'App\Models\Ad') {
                            return $record->modifiable->title ?? '';
                        }
                        if ($record->modifiable_type == 'App\Models\Media') {
                            return $record->modifiable->model->title ?? '';
                        }
                        return '';
                    }),
                TextColumn::make('user')
                    ->label(__('messages.t_ap_user'))
                    ->default(fn($record) => $record->modifier->name ?? ''),
                TextColumn::make('field')
                    ->label(__('messages.t_ap_field'))
                    ->default(function ($record) {
                        if ($record->modifiable_type == 'App\Models\Ad') {
                            return array_key_first($record->modifications) ?? '';
                        }
                        if ($record->modifiable_type == 'App\Models\Media') {
                            return 'image' ?? '';
                        }
                        return '';
                    }),
                TextColumn::make('modified')
                    ->label(__('messages.t_ap_modified'))
                    ->limit(35)
                    ->default(function ($record) {
                        $flattened = \Arr::dot($record->modifications);
                        return $record->modifiable_type == 'App\Models\Ad' ? reset($flattened) : '';
                    }),
                TextColumn::make('original')
                    ->label(__('messages.t_ap_original'))
                    ->limit(35)
                    ->default(function ($record) {
                        $flattened = \Arr::dot($record->modifications);
                        return $record->modifiable_type == 'App\Models\Ad' ? end($flattened) : '';
                    }),
                ImageColumn::make('image')
                    ->label(__('messages.t_ap_image'))
                    ->defaultImageUrl(function ($record) {
                        return $record->modifiable_type == 'App\Models\Media' ? $record->modifiable?->getUrl() : '';
                    })
                // ->extraAttributes(fn($record)=>$record->modifiable->getUrl()?['class'=>'hidden']:[])

            ])
            ->filters([
                //
            ])
            ->actions([
                Action::make('approve')
                    ->label(__('messages.t_ap_approve'))
                    ->icon('heroicon-m-check-circle')
                    ->color('success')
                    ->action(function ($record, $data) {
                        $reason = $data['reason'] ?? '';
                        try {
                            $modification = $record;
                            auth()->user()->approve($record, $reason);

                            if ($modification->modifiable_type == 'App\Models\Ad') {
                                $modificationField = $modification->modifications;
                                if (array_key_exists('title', $modificationField) && $modification->modifiable && $modification->modifiable->title) {
                                    self::updateAdSlug($modification->modifiable, $modification->modifiable->title);
                                }
                            }
                            Notification::make()
                                ->title(__('messages.t_ap_modification_request_rejected'))
                                ->success()
                                ->send();
                        } catch (\Exception $e) {
                            Notification::make()
                                ->title($e->getMessage())
                                ->danger()
                                ->send();
                        }
                    }),
                Action::make('reject')
                    ->label(__('messages.t_ap_reject'))
                    ->color('danger')
                    ->icon('heroicon-m-x-mark')
                    ->form([
                        Textarea::make('reason')
                            ->label(__('messages.t_ap_reason'))
                    ])
                    ->action(function ($record, $data) {
                        $reason = $data['reason'] ?? '';
                        try {
                            if ($record->modifiable_type == 'App\Models\Media') {
                                if (isset($data['reason'])) {
                                    $disapprovalModel = config('approval.models.disapproval', \Approval\Models\Disapproval::class);
                                    $disapprovalModel::firstOrCreate([
                                        'disapprover_id' => auth()->id(),
                                        'disapprover_type' => 'App\Models\User',
                                        'modification_id' => $record->id,
                                        'reason' => $reason
                                    ]);
                                }
                                $record->active = false;
                                $record->save();
                            }
                            if ($record->modifiable_type == 'App\Models\Ad') {
                                auth()->user()->disapprove($record, $reason);
                            }
                            Notification::make()
                                ->title(__('messages.t_ap_modification_request_accepted'))
                                ->success()
                                ->send();
                        } catch (\Exception $e) {
                            Notification::make()
                                ->title($e->getMessage())
                                ->danger()
                                ->send();
                        }

                    }),
                ViewAction::make('view')
                    ->infolist([
                        Grid::make([
                            'default' => 2
                        ])
                            ->schema([
                                TextEntry::make('ad')
                                    ->label(__('messages.t_ap_ad'))
                                    ->default(function ($record) {
                                        if ($record->modifiable_type == 'App\Models\Ad') {
                                            return $record->modifiable->title ?? '';
                                        }
                                        if ($record->modifiable_type == 'App\Models\Media') {
                                            return $record->modifiable->model->title ?? '';
                                        }
                                        return '';
                                    }),
                                TextEntry::make('user')
                                    ->label(__('messages.t_ap_user'))
                                    ->default(fn($record): string => $record->modifier->name ?? ''),
                                TextEntry::make('field')
                                    ->label(__('messages.t_ap_field'))
                                    ->default(function ($record): string {
                                        if ($record->modifiable_type == 'App\Models\Ad') {
                                            return array_key_first($record->modifications) ?? '';
                                        }
                                        if ($record->modifiable_type == 'App\Models\Media') {
                                            return 'image' ?? '';
                                        }
                                        return '';
                                    }),
                                TextEntry::make('modified')
                                    ->label(__('messages.t_ap_modified'))
                                    ->columnSpanFull()
                                    ->default(function ($record): string {
                                        $flattened = \Arr::dot($record->modifications);
                                        return $record->modifiable_type == 'App\Models\Ad' ? reset($flattened) : '';
                                    }),
                                TextEntry::make('original')
                                    ->label(__('messages.t_ap_original'))
                                    ->columnSpanFull()
                                    ->default(function ($record): string {
                                        $flattened = \Arr::dot($record->modifications);
                                        return $record->modifiable_type == 'App\Models\Ad' ? end($flattened) : '';
                                    }),
                                ImageEntry::make('image')
                                    ->label(__('messages.t_ap_image'))
                                    ->columnSpanFull()
                                    ->defaultImageUrl(function ($record) {
                                        return $record->modifiable_type == 'App\Models\Media' ? $record->modifiable?->getUrl() : '';
                                    })
                            ])
                    ])
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    protected static function updateAdSlug($ad, $title)
    {
        $ad->slug = Str::slug(Str::limit($title, 138)) . '-' . substr($ad->id, 0, 8);
        $ad->save();
    }
    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListModifications::route('/'),
            // 'create' => Pages\CreateModification::route('/create'),
            // 'edit' => Pages\EditModification::route('/{record}/edit'),
        ];
    }
}
