<?php

namespace App\Filament\Resources;

use App\Filament\Clusters\AdManagement;
use App\Filament\Resources\AdTypeResource\Pages;
use App\Models\AdType;
use App\Models\City;
use App\Models\Country;
use App\Models\PriceType;
use App\Models\State;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Components\Fieldset;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Support\Str;

class AdTypeResource extends Resource
{
    protected static ?string $model = AdType::class;
    protected static ?string $cluster = AdManagement::class;
    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function canAccess(): bool
    {
        return userHasPermission('view_ad::type');
    }

    public static function canCreate(): bool
    {
        return userHasPermission('create_ad::type');
    }

    public static function canEdit($record): bool
    {
        return userHasPermission('edit_ad::type');
    }

    public static function canDelete($record): bool
    {
        return userHasPermission('delete_ad::type');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                self::getBasicInformation(),
                self::getMarketplaceConfiguration(),
                self::getAdConfiguration(),
                self::getAdditionalOptions(),
                self::getFilterConfiguration(),
                // self::getImageConfiguration(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')->searchable(),
                // ToggleColumn::make('is_default')
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()->hidden(fn($record) => $record->is_default),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getBasicInformation()
    {
        $fileUpload = SpatieMediaLibraryFileUpload::make('icon')
            ->maxSize(maxUploadFileSize())
            ->collection('ad_type_icons')
            ->visibility('public')
            ->image()
            ->required()
            ->imageEditor()
            ->avatar();

        $storageType = config('filesystems.default');

        if ($storageType == 's3') {
            $fileUpload->disk($storageType);
        }

        return Section::make(__('messages.t_ad_type_basic_information'))
            ->schema([
                $fileUpload->helperText(__('messages.t_ad_type_icon_helper_text')),

                TextInput::make('name')
                    ->label(__('messages.t_ad_type_name_label'))
                    ->required()
                    ->live(onBlur: true)
                    ->afterStateUpdated(fn(callable $get, callable $set) => $set('slug', Str::slug($get('name'))))
                    ->helperText(__('messages.t_ad_type_name_helper_text'))
                    ->columnSpan(2),

                TextInput::make('slug')
                    ->unique(AdType::class, 'slug', ignoreRecord: true)
                    ->afterStateUpdated(fn($state) => Str::slug($state))
                    ->required()
                    ->hintAction(
                        Action::make('generateSlug')
                            ->label(__('messages.t_ad_type_generate_slug'))
                            ->icon('heroicon-m-bolt')
                            ->action(function (Set $set, Get $get) {
                                if ($get('name')) {
                                    $set('slug', Str::slug($get('name')));
                                } else {
                                    $set('slug', Str::slug(Str::random(10)));
                                }
                            })
                    )
                    ->helperText(__('messages.t_ad_type_slug_helper_text'))
                    ->columnSpan(2),

                ColorPicker::make('color')
                    ->label(__('messages.t_ad_type_color_label'))
                    ->nullable()
                    ->helperText(__('messages.t_ad_type_color_helper_text')),
            ])->columns(3);
    }

    public static function getAdConfiguration()
    {
        return Section::make(__('messages.t_ad_type_ad_configuration'))
            ->columns(3)
            ->schema([
                Fieldset::make(__('messages.t_ad_type_default_fields'))
                    ->schema([
                        Toggle::make('enable_title')
                            ->label(__('messages.t_ad_type_enable_title'))
                            ->default(true)
                            ->disabled()
                            ->helperText(__('messages.t_ad_type_enable_title_helper')),

                        Toggle::make('enable_description')
                            ->label(__('messages.t_ad_type_enable_description'))
                            ->default(true)
                            ->disabled()
                            ->helperText(__('messages.t_ad_type_enable_description_helper')),
                    ])->columns(2),

                Fieldset::make(__('messages.t_ad_type_price_settings'))
                    ->schema([
                        Toggle::make('enable_price')
                            ->label(__('messages.t_ad_type_enable_price'))
                            ->default(true)
                            ->visible(fn(callable $get) => $get('marketplace') != CLASSIFIED_MARKETPLACE)
                            ->helperText(__('messages.t_ad_type_enable_price_helper')),

                        Toggle::make('enable_offer_price')
                            ->label(__('messages.t_ad_type_enable_offer_price'))
                            ->default(true)
                            // ->visible(fn(callable $get) => $get('marketplace') != CLASSIFIED_MARKETPLACE)
                            ->helperText(__('messages.t_ad_type_enable_offer_price_helper')),

                        Toggle::make('disable_price_type')
                            ->helperText(__('messages.t_ad_type_disable_price_type_helper'))
                            ->live(onBlur: true)
                            ->afterStateUpdated(function (Set $set, $state) {
                                $set('customize_price_type', false);
                                $set('enable_price', $state);
                            })->visible(fn(callable $get) => !in_array($get('marketplace'), [ONLINE_SHOP_MARKETPLACE, POINT_SYSTEM_MARKETPLACE])),

                        Toggle::make('customize_price_type')
                            ->helperText(__('messages.t_ad_type_customize_price_type_helper'))
                            ->afterStateUpdated(function (Set $set) {
                                $set('disable_price_type', false);
                            })->visible(fn(callable $get) => !in_array($get('marketplace'), [ONLINE_SHOP_MARKETPLACE, POINT_SYSTEM_MARKETPLACE]))
                            ->live(onBlur: true),

                        Select::make('price_types')
                            ->multiple()
                            ->helperText(__('messages.t_ad_type_price_types_helper'))
                            ->required(fn(Get $get) => $get('customize_price_type'))
                            ->hidden(fn(Get $get) => !$get('customize_price_type'))
                            ->options(PriceType::all()->pluck('name', 'id'))
                            ->visible(fn(callable $get) => !in_array($get('marketplace'), [ONLINE_SHOP_MARKETPLACE, POINT_SYSTEM_MARKETPLACE])),

                        TagsInput::make('field_options')
                            ->label(__('messages.t_ad_type_field_options'))
                            ->live(onBlur: true)
                            ->visible(fn($get) => $get('field_type') == 'select')
                            ->helperText(__('messages.t_ad_type_field_options_helper')),

                        Toggle::make('has_price_suffix')
                            ->hidden(fn(Get $get) => $get('disable_price_type'))
                            ->helperText(__('messages.t_ad_type_has_price_suffix_helper'))
                            ->live(onBlur: true),

                        TagsInput::make('suffix_field_options')
                            ->placeholder(__('messages.t_ad_type_placeholder_suffix_options'))
                            ->hidden(fn(Get $get) => $get('disable_price_type'))
                            ->helperText(__('messages.t_ad_type_suffix_field_options_helper'))
                            ->required(fn(Get $get) => $get('has_price_suffix')),
                    ])->columns(3)->visible(fn(callable $get) => $get('marketplace') != POINT_SYSTEM_MARKETPLACE),

                Fieldset::make(__('messages.t_ad_type_location_settings'))
                    ->schema([
                        Toggle::make('disable_location')
                            ->helperText(__('messages.t_ad_type_disable_location_helper'))
                            ->reactive()
                            ->afterStateUpdated(fn($state, Set $set) => $state ? $set('default_location', false) : null),

                        Toggle::make('default_location')
                            ->label(__('messages.t_ad_type_default_location'))
                            ->live(onBlur: true)
                            ->default(false)
                            ->helperText(__('messages.t_ad_type_default_location_helper')),

                        Section::make(__('messages.t_ad_type_location_details'))
                            ->collapsible()
                            ->visible(fn(Get $get) => $get('default_location'))
                            ->schema([
                                Select::make('country_id')
                                    ->label(__('messages.t_ad_type_country'))
                                    ->options(Country::pluck('name', 'id')->toArray())
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(fn(callable $set) => $set('state_id', null))
                                    ->required(),

                                Select::make('state_id')
                                    ->label(__('messages.t_ad_type_state'))
                                    ->options(function (Get $get) {
                                        $countryId = $get('country_id');
                                        if (!$countryId) {
                                            return [];
                                        }
                                        return State::where('country_id', $countryId)->pluck('name', 'id')->toArray();
                                    })
                                    ->live(onBlur: true)
                                    ->hidden(fn(Get $get): bool => !$get('country_id'))
                                    ->afterStateUpdated(fn(callable $set) => $set('city_id', null))
                                    ->required(),

                                Select::make('city_id')
                                    ->label(__('messages.t_ad_type_city'))
                                    ->options(function (Get $get) {
                                        $stateId = $get('state_id');
                                        if (!$stateId) {
                                            return [];
                                        }
                                        return City::where('state_id', $stateId)->pluck('name', 'id')->toArray();
                                    })
                                    ->hidden(fn(Get $get): bool => !$get('state_id'))
                                    ->required(),
                            ])
                            ->columnSpanFull(),
                    ])->columns(3),
            ]);
    }

    public static function getAdditionalOptions()
    {
        return Section::make(__('messages.t_ad_type_additional_options'))
            ->columns(3)
            ->schema([
                Toggle::make('enable_tags')
                    ->label(__('messages.t_ad_type_enable_tags'))
                    ->default(false)
                    ->helperText(__('messages.t_ad_type_enable_tags_helper')),

                Toggle::make('allowed_comment')
                    ->label(__('messages.t_ad_type_allow_comments'))
                    ->default(false)
                    ->helperText(__('messages.t_ad_type_allow_comments_helper')),

                Toggle::make('enable_for_sale_by')
                    ->label(__('messages.t_ad_type_enable_for_sale_by'))
                    ->default(false)
                    ->helperText(__('messages.t_ad_type_enable_for_sale_by_helper')),

                Toggle::make('disable_condition')
                    ->label(__('messages.t_ad_type_disable_condition'))
                    ->default(false)
                    ->helperText(__('messages.t_ad_type_disable_condition_helper')),
            ]);
    }

    public static function getFilterConfiguration()
    {
        return Section::make(__('messages.t_ad_type_filter_configuration'))
            ->columns(3)
            ->schema([
                Toggle::make('enable_filters')
                    ->label(__('messages.t_ad_type_enable_filters')),

                Repeater::make('filter_options')
                    ->label('')
                    ->schema([
                        Toggle::make('enable_categories')
                            ->label(__('messages.t_ad_type_enable_categories'))
                            ->helperText(__('messages.t_ad_type_enable_categories_helper')),

                        Toggle::make('enable_sort_by')
                            ->label(__('messages.t_ad_type_enable_sort_by'))
                            ->helperText(__('messages.t_ad_type_enable_sort_by_helper')),

                        Toggle::make('enable_date_range')
                            ->label(__('messages.t_ad_type_enable_date_range_filter'))
                            ->live(onBlur: true)
                            ->helperText(__('messages.t_ad_type_enable_date_range_filter_helper')),

                        Toggle::make('enable_price_range')
                            ->label(__('messages.t_ad_type_enable_price_range_filter'))
                            ->live(onBlur: true)
                            ->helperText(__('messages.t_ad_type_enable_price_range_filter_helper')),

                        Toggle::make('enable_price_range_toggle')
                            ->label(__('messages.t_ad_type_enable_price_range_toggle_filter'))
                            ->visible(fn(callable $get) => $get('enable_price_range'))
                            ->helperText(__('messages.t_ad_type_enable_price_range_toggle_filter_helper')),
                    ])
                    ->columns(2)
                    ->deletable(false)
                    ->addable(false)
                    ->columnSpan(2)
                    ->reorderable(false),
            ]);
    }

    public static function getMarketplaceConfiguration()
    {
        return Section::make(__('messages.t_ad_type_marketplace_configuration'))
            ->columns(2)
            ->schema([
                Select::make('marketplace')
                    ->options([
                        'classified' =>  __('messages.t_ad_type_classified'),
                        ...is_ecommerce_active() ? ['online_shop' =>  __('messages.t_ad_type_online_shop')] : [],
                        ...isVehicleRentalMarketplacePluginEnabled() ? ['vehicle_rental' =>  __('messages.t_ad_type_vehicle_rental')] : [],
                        ...isEnablePointSystem() ? ['point_system' =>  __('messages.t_ad_type_point_system')] : []
                    ])
                    ->required()
                    ->default('classified')
                    ->live(onBlur: true)
                    ->hintAction(
                        Action::make('Adjust the global configuration settings')
                            ->label(__('messages.t_ad_type_adjust_global_configuration_action'))
                            ->icon('heroicon-m-link')
                            ->url(function ($state) {
                                if (ONLINE_SHOP_MARKETPLACE == $state) {
                                    return route('filament.admin.ecommerce.pages.e-commerce-configuration');
                                } elseif (VEHICLE_RENTAL_MARKETPLACE == $state) {
                                    return '/admin/vehicle-rental/manage-vehicle-rental-settings';
                                } elseif (POINT_SYSTEM_MARKETPLACE == $state) {
                                    return '/admin/point-setting/point-vault-configuration';
                                }
                            })
                            ->openUrlInNewTab()
                    ),

                Repeater::make('marketplace_options')
                    ->label('')
                    ->schema(function (callable $get) {
                        return self::getMarketplaceOptions($get);
                    })
                    ->columns(2)
                    ->deletable(false)
                    ->addable(false)
                    ->columnSpan(2)
                    ->visible(fn(callable $get) => in_array($get('marketplace'), [ONLINE_SHOP_MARKETPLACE]))
                    ->reorderable(false),
            ]);
    }

    public static function getMarketplaceOptions($get)
    {
        if ($get('marketplace') == ONLINE_SHOP_MARKETPLACE) {
            return self::getOnlineShopOptions();
        } elseif ($get('marketplace') == VEHICLE_RENTAL_MARKETPLACE) {
            return self::getVehicleRentalOptions();
        } elseif ($get('marketplace') == POINT_SYSTEM_MARKETPLACE) {
            return self::getOnlineShopOptions();
        } else {
            return self::getClassifiedOptions();
        }
    }

    public static function getClassifiedOptions()
    {
        return [
            Toggle::make('enable_price_type')
                ->label(__('messages.t_ad_type_enable_price_type'))
                ->helperText(__('messages.t_ad_type_enable_price_type_helper'))
                ->default(false),

            Toggle::make('enable_price_suffix')
                ->label(__('messages.t_ad_type_enable_price_suffix'))
                ->helperText(__('messages.t_ad_type_enable_price_suffix_helper'))
                ->default(false),

            Toggle::make('enable_condition')
                ->label(__('messages.t_ad_type_enable_condition'))
                ->helperText(__('messages.t_ad_type_enable_condition_helper'))
                ->default(false),
        ];
    }

    public static function getOnlineShopOptions()
    {
        return [
            Toggle::make('enable_sku')
                ->label(__('messages.t_ad_type_enable_sku'))
                ->helperText(__('messages.t_ad_type_enable_sku_helper')),

            Toggle::make('disable_cash_on_delivery')
                ->label(__('messages.t_ad_type_disable_cash_on_delivery'))
                ->helperText(__('messages.t_ad_type_disable_cash_on_delivery_helper')),
        ];
    }

    public static function getVehicleRentalOptions()
    {
        return [
            Toggle::make('enable_make')
                ->label('Enable Make') // Label
                ->helperText('Allow users to specify the vehicle make (e.g., Toyota, Ford).'),

            Toggle::make('enable_model')
                ->label('Enable Model') // Label
                ->helperText('Allow users to specify the vehicle model (e.g., Corolla, Mustang).'),

            Toggle::make('enable_availability_window')
                ->label('Enable Availability Window (days)') // Label
                ->helperText('Allow users to define the number of days the vehicle is available for rent.'),

            Toggle::make('enable_trip_length')
                ->label('Enable Minimum and Maximum Trip Length (days)') // Label
                ->helperText('Allow users to specify the minimum and maximum number of rental days.'),

            Toggle::make('enable_transmission')
                ->label('Enable Transmission Type') // Label
                ->helperText('Allow users to specify the vehicle’s transmission type (e.g., automatic, manual).'),

            Toggle::make('enable_fuel_type')
                ->label('Display Fuel Type') // Label
                ->helperText('Allow users to display the type of fuel the vehicle uses (e.g., petrol, diesel, electric).'),

            Toggle::make('enable_mileage')
                ->label('Display Mileage (km)') // Label
                ->helperText('Allow users to display the mileage of the vehicle in kilometers.'),

            Toggle::make('enable_start_date')
                ->label('Enable Start Date') // Label
                ->helperText('Allow users to specify the rental start date.'),

            Toggle::make('enable_end_date')
                ->label('Enable End Date') // Label
                ->helperText('Allow users to specify the rental end date.'),
        ];
    }


    public static function getImageConfiguration()
    {
        return Section::make(__('messages.form.sections.image_configuration'))
            ->columns(4)
            ->schema([
                Toggle::make('allowed_upload_image')
                    ->default(false)
                    ->label(__('messages.form.labels.enable_image'))
                    ->live(onBlur: true)
                    ->helperText(__('messages.form.helpers.enable_image')),

                Repeater::make('upload_image_options')
                    ->label('')
                    ->schema([
                        Fieldset::make('Basic Settings')
                            ->schema([
                                Toggle::make('multiple')
                                    ->label('Allow Multiple Files')
                                    ->default(false)
                                    ->live(onBlur: true)
                                    ->helperText('Enable or disable the ability to upload multiple files at once.'),

                                TextInput::make('upload_directory')
                                    ->label('Upload Directory')
                                    ->default('uploads')
                                    ->required()
                                    ->helperText('Specify the directory where files will be uploaded.'),

                                Toggle::make('required')
                                    ->label('Required Field')
                                    ->default(false)
                                    ->helperText('Mark this field as required.'),

                                TextInput::make('min_size')
                                    ->label('Minimum File Size (KB)')
                                    ->numeric()
                                    ->required()
                                    ->default(1)
                                    ->helperText('Specify the minimum file size in kilobytes.'),

                                TextInput::make('max_size')
                                    ->label('Maximum File Size (KB)')
                                    ->numeric()
                                    ->required()
                                    ->default(5120)
                                    ->helperText('Specify the maximum file size in kilobytes.'),

                                TextInput::make('min_files')
                                    ->label('Minimum Number of Files')
                                    ->numeric()
                                    ->required()
                                    ->default(1)
                                    ->visible(fn(Get $get) => $get('multiple'))
                                    ->helperText('Set the minimum number of files to be uploaded when multiple uploads are allowed.'),

                                TextInput::make('max_files')
                                    ->label('Maximum Number of Files')
                                    ->numeric()
                                    ->required()
                                    ->default(10)
                                    ->visible(fn(Get $get) => $get('multiple'))
                                    ->helperText('Set the maximum number of files allowed for upload when multiple uploads are enabled.'),
                            ])
                            ->columns(3),

                        Fieldset::make('File Type Settings')
                            ->schema([
                                Toggle::make('is_image')
                                    ->label('Image Upload Only')
                                    ->default(false)
                                    ->live(onBlur: true)
                                    ->helperText('Enable or disable uploading of non-image files.'),

                                Select::make('accepted_file_types')
                                    ->label('Accepted File Types')
                                    ->multiple()
                                    ->options([
                                        'image/jpeg' => 'JPEG',
                                        'image/png' => 'PNG',
                                        'image/gif' => 'GIF',
                                        'image/bmp' => 'BMP',
                                        'image/svg+xml' => 'SVG',
                                        'application/pdf' => 'PDF',
                                        'application/msword' => 'DOC',
                                        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'DOCX',
                                        'text/plain' => 'TXT',
                                    ])
                                    ->hidden(fn(Get $get) => $get('is_image'))
                                    ->helperText('Select which file types are allowed for upload.'),
                            ])
                            ->columns(2),

                        Fieldset::make('Image Settings')
                            ->schema([
                                Toggle::make('image_editor')
                                    ->label('Enable Image Editor')
                                    ->default(false)
                                    ->live(onBlur: true)
                                    ->visible(fn(Get $get) => $get('is_image'))
                                    ->helperText('Allow users to edit images after uploading.'),

                                Toggle::make('image_crop_aspect_ratio')
                                    ->label('Enable Crop Aspect Ratio')
                                    ->default(false)
                                    ->live(onBlur: true)
                                    ->visible(fn(Get $get) => $get('is_image') && $get('image_editor'))
                                    ->helperText('Enable cropping images with a specific aspect ratio.'),

                                TextInput::make('image_crop_aspect_ratio_width')
                                    ->label('Crop Aspect Ratio Width')
                                    ->numeric()
                                    ->visible(fn(Get $get) => $get('is_image') && $get('image_editor') && $get('image_crop_aspect_ratio'))
                                    ->helperText('Specify the width of the crop aspect ratio.'),

                                TextInput::make('image_crop_aspect_ratio_height')
                                    ->label('Crop Aspect Ratio Height')
                                    ->numeric()
                                    ->visible(fn(Get $get) => $get('is_image') && $get('image_editor') && $get('image_crop_aspect_ratio'))
                                    ->helperText('Specify the height of the crop aspect ratio.'),

                                Select::make('image_resize_mode_option')
                                    ->label('Image Resize Mode')
                                    ->options([
                                        'cover' => 'Cover',
                                        'contain' => 'Contain',
                                        'stretch' => 'Stretch',
                                    ])
                                    ->default('cover')
                                    ->visible(fn(Get $get) => $get('is_image'))
                                    ->helperText('Select how images should be resized.'),

                                Toggle::make('should_maintain_proportion')
                                    ->label('Maintain Image Proportion')
                                    ->default(true)
                                    ->visible(fn(Get $get) => $get('is_image'))
                                    ->helperText('Ensure image proportions are maintained during resizing.'),
                            ])
                            ->columns(3)
                            ->visible(fn(Get $get) => $get('is_image')),

                        Fieldset::make('Display Options')
                            ->schema([
                                Toggle::make('downloadable')
                                    ->label('Allow Downloads')
                                    ->default(true)
                                    ->helperText('Enable or disable the option for users to download images.'),

                                Toggle::make('open_in_new_tab')
                                    ->label('Open in New Tab')
                                    ->default(true)
                                    ->helperText('Specify if images should open in a new tab when clicked.'),

                                Toggle::make('reorderable')
                                    ->label('Allow Reordering')
                                    ->default(false)
                                    ->visible(fn(Get $get) => $get('multiple'))
                                    ->helperText('Enable or disable the option to reorder multiple files.'),

                                Toggle::make('readable_size')
                                    ->label('Show Readable Size')
                                    ->default(true)
                                    ->helperText('Display the readable file size next to images.'),

                                Toggle::make('show_dimensions')
                                    ->label('Show Image Dimensions')
                                    ->default(true)
                                    ->visible(fn(Get $get) => $get('is_image'))
                                    ->helperText('Display the dimensions of images (width x height).'),
                            ])
                            ->columns(3),

                        Fieldset::make('Storage Settings')
                            ->schema([
                                TextInput::make('disk_name')
                                    ->label('Storage Disk')
                                    ->default('public')
                                    ->helperText('Specify the storage disk for file storage.'),
                            ])
                            ->columns(2),

                        Fieldset::make('Messages')
                            ->schema([
                                TextInput::make('upload_error_message')
                                    ->label('Upload Error Message')
                                    ->default('Failed to upload file.')
                                    ->helperText('Message displayed when file upload fails.'),

                                TextInput::make('remove_error_message')
                                    ->label('Remove Error Message')
                                    ->default('Failed to remove file.')
                                    ->helperText('Message displayed when file removal fails.'),

                                TextInput::make('upload_progress_message')
                                    ->label('Upload Progress Message')
                                    ->default('Uploading file...')
                                    ->helperText('Message displayed during file upload progress.'),
                            ])
                            ->columns(3),
                    ])->columns(2)
                    ->deletable(false)
                    ->addable(false)
                    ->columnSpan(3)
                    ->reorderable(false),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListAdTypes::route('/'),
            'create' => Pages\CreateAdType::route('/create'),
            'edit' => Pages\EditAdType::route('/{record}/edit'),
        ];
    }
}
