<?php

namespace App\Foundation\AdBase\Traits;

use App\Settings\PointVaultSettings;
use App\Models\Ad;
use App\Models\AdCondition;
use App\Models\AdType;
use App\Models\PriceType;
use App\Settings\GeneralSettings;
use App\Settings\OfflinePaymentSettings;
use Closure;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\MarkdownEditor;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\ToggleButtons;
use Filament\Forms\Get;
use Ysfkaya\FilamentPhoneInput\Forms\PhoneInput;
use Filament\Forms\Set;
use App\Settings\PhoneSettings;

trait AdMethods
{
    abstract public function adTypeSelect();
    abstract public function titleInput();
    abstract public function detailForm();
    abstract public function updated($name, $value);
    abstract public function save();
    abstract public function checkRequiredFieldsFilled();

    public function getCategoryId()
    {
        return Hidden::make('category_id')
            ->live();
        // TODO: Handle reset logic when category changes
        // ->afterStateUpdated(function(Set $set){
        //     $set('condition_id', null);
        //     $set('price_type_id', null);
        //     $set('price', null);
        // });
    }

    public function getForSaleByToggle()
    {
        return ToggleButtons::make('for_sale_by')
            ->label(__('messages.t_for_sale_by'))
            ->live()
            ->grouped()
            ->visible($this->validateAdTypePresence('enable_for_sale_by'))
            ->options([
                'owner' => __('messages.t_owner_for_sale'),
                'business' => __('messages.t_business_for_sale'),
            ]);
    }

    public function getDescription()
    {
        return MarkdownEditor::make('description')
            ->disableToolbarButtons(['attachFiles'])
            ->label(__('messages.t_description'))
            ->live(onBlur: true)
            ->minLength(20)
            ->required();
    }

    public function getConditionToggle()
    {
        return ToggleButtons::make('condition_id')
            ->hidden($this->validateAdTypePresence('disable_condition'))
            ->label(__('messages.t_condition'))
            ->live()
            ->options(AdCondition::all()->pluck('name', 'id'))
            ->inline();
    }

    public function getPriceTypeSelect()
    {
        return Select::make('price_type_id')
            ->hidden($this->hasEnableOnlineShopping() || $this->validateAdTypePresence('disable_price_type') || isEnablePointSystem())
            ->selectablePlaceholder(false)
            ->label(__('messages.t_price_type'))
            ->live()
            ->required()
            ->native(false)
            ->options(function () {
                $priceTypeIds = $this->validateAdTypePresence('price_types');

                return PriceType::when($this->validateAdTypePresence('customize_price_type') && count($priceTypeIds) > 0, function ($query) use ($priceTypeIds) {
                    $query->whereIn('id', $priceTypeIds);
                })->pluck('name', 'id');
            })->columnSpanFull();
    }

    public function getPriceSuffixSelect()
    {
        return Select::make('price_suffix')
            ->default('days')
            ->label(__('messages.t_price_suffix'))
            ->visible(function(Get $get,$livewire) {
                return  $get('price_type_id') == 1 && $this->validateAdTypePresence('has_price_suffix');
            })
            ->required()
            ->helperText(__('messages.t_price_suffix_helper'))
            ->default('day')
            ->options(function(Get $get){
                $adType=AdType::find($get('ad_type_id'));
                return array_combine($adType->suffix_field_options ?? [], $adType->suffix_field_options ?? []);
            });
    }

    public function getPriceInput()
    {
        return TextInput::make('price')
            ->helperText(__('messages.t_set_fair_price'))
            ->required()
            ->validationAttribute(__('messages.t_price'))
            ->numeric()
            ->minValue(1)
            ->placeholder(__('messages.t_price_your_ad'))
            ->prefix(config('app.currency_symbol'))
            ->live(onBlur: true)
            ->maxValue(fn() => app(PointVaultSettings::class)->set_max_points_ad)
            ->columnSpanFull(isEnablePointSystem())
            ->hidden(function (Get $get) {
                if ($this->hasEnableOnlineShopping() || isEnablePointSystem())  return false;

                if ($this->validateAdTypePresence('disable_price_type') || $get('price_type_id') != 1 || $this->validateAdTypePresence('has_price_suffix')) {
                    return true;
                }
            })
            ->hiddenLabel(function (Get $get) {
                return !$this->validateAdTypePresence('enable_price');
            });
    }

    public function getOfferPriceInput()
    {
        return TextInput::make('offer_price')
            ->label(__('messages.t_offer_price'))
            ->helperText(__('messages.t_set_offer_price'))
            ->validationAttribute(__('messages.t_offer_price'))
            ->numeric()
            ->minValue(1)
            ->lt('price')
            ->placeholder(__('messages.t_enter_offer_price'))
            ->prefix(config('app.currency_symbol'))
            ->live(onBlur: true)
            ->hidden(function (Get $get) {
                if ($this->hasEnableOnlineShopping() && $this->validateAdTypePresence('enable_offer_price'))  return false;

                if (!$this->validateAdTypePresence('enable_offer_price') || $this->validateAdTypePresence('disable_price_type') || $get('price_type_id') != 1 || isEnablePointSystem()) {
                    return true;
                }
            });
    }

    public function getDisplayPhoneToggle()
    {
        return ToggleButtons::make('display_phone')
            ->label(__('messages.t_display_phone_number'))
            ->live()
            ->boolean()
            ->grouped()
            ->visible(app(PhoneSettings::class)->enable_phone)
            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                // Update phone_number based on display_phone toggle
                if ($state) {
                    $this->phone_number = $this->ad?->phone_number ? $this->ad?->phone_number : auth()->user()?->phone_number;
                    $this->ad->phone_number = $this->phone_number;
                    $this->ad->save();
                    $this->checkRequiredFieldsFilled();
                    $set('phone_number', $this->phone_number);
                }
            });
    }

    public function getPhoneNumberInput()
    {
        return PhoneInput::make('phone_number')
            ->initialCountry(app(GeneralSettings::class)->default_mobile_country ?? 'us')
            ->placeholder(__('messages.t_enter_phone_number'))
            ->helperText(__('messages.t_phone_number_helper'))
            ->required()
            ->live()
            ->visible(app(PhoneSettings::class)->enable_phone)
            ->hidden(fn(Get $get): bool => !$get('display_phone'));
    }

    public function getSameNumberToggle()
    {
        return ToggleButtons::make('display_whatsapp')
            ->label(__('messages.t_display_whatsapp_number'))
            ->afterStateUpdated(function ($state, Get $get, Set $set) {
                // Update whatsapp_number when display_whatsapp is toggled
                if ($state) {
                    $this->whatsapp_number = $this->ad->whatsapp_number ? $this->ad->whatsapp_number : $get('phone_number');
                    $this->ad->whatsapp_number = $this->whatsapp_number;
                    $this->ad->save();
                    $this->checkRequiredFieldsFilled();
                    $set('whatsapp_number', $this->whatsapp_number);
                }
            })
            ->live()
            ->boolean()
            ->visible(app(PhoneSettings::class)->enable_phone)
            ->grouped();
    }

    public function getWhatsappNumberInput()
    {
        return PhoneInput::make('whatsapp_number')
            ->initialCountry(app(GeneralSettings::class)->default_mobile_country ?? 'us')
            ->placeholder(__('messages.t_enter_phone_number'))
            ->helperText(__('messages.t_display_whatsapp_helper_text'))
            ->live()
            ->required()
            ->visible(app(PhoneSettings::class)->enable_phone)
            ->hidden(fn(Get $get): bool => !$get('display_whatsapp'));
    }

    public function getTagsInput()
    {
        return TagsInput::make('tags')
            ->label(__('messages.t_tags'))
            ->helperText(__('messages.t_set_tags'))
            ->visible($this->validateAdTypePresence('enable_tags'))
            ->live(onBlur: true);
    }

    public function createSkuInput()
    {
        return TextInput::make('sku')
            ->rules([
                fn(Get $get): Closure => function (string $attribute, $value, Closure $fail) use ($get) {
                    $ad = Ad::where('sku', $value)->exists();
                    if ($ad && $this->ad->sku != $value) {
                        $fail("The {$attribute} already exists.");
                    }
                },
            ])
            ->maxLength(20)
            ->label(__('messages.t_sku'))
            ->helperText(__('messages.t_sku_helper_text'))
            ->required()
            ->validationAttribute(__('messages.t_sku'))
            ->minLength(5)
            ->placeholder(__('messages.t_sku'))
            ->visible($this->hasEnableOnlineShopping() && $this->validateAdTypePresence('marketplace_settings')['enable_sku'] ?? false)
            ->live(onBlur: true);
    }

    public function createReturnPolicySelect()
    {
        return Select::make('return_policy_id')
            ->label(__('messages.t_return_policy'))
            ->helperText(__('messages.t_sku_helper_text'))
            ->required()
            ->live(onBlur: true)
            ->visible($this->hasEnableOnlineShopping())
            ->options(\Adfox\ECommerce\Models\ReturnPolicy::where('user_id', auth()->id())->pluck('policy_name', 'id'));
    }

    public function createCashOnDeliveryToggle()
    {
        return ToggleButtons::make('enable_cash_on_delivery')
            ->label(__('messages.t_enable_cash_on_delivery'))
            ->helperText(__('messages.t_enable_cash_on_delivery_helper_text'))
            ->default(1)
            ->grouped()
            ->boolean()
            // ->default(($this->hasEnableOnlineShopping() && ($this->validateAdTypePresence('marketplace_settings')['disable_cash_on_delivery'] ?? false)) ? true : false)
            ->visible(app('filament')->hasPlugin('offline-payment') && app(OfflinePaymentSettings::class)->status && ($this->validateAdTypePresence('marketplace_settings')['disable_cash_on_delivery'] ?? false))
            ->validationAttribute(__('messages.t_enable_cash_on_delivery'));
    }

    public function hasEnableOnlineShopping()
    {
        return ($this->validateAdTypePresence('marketplace') == ONLINE_SHOP_MARKETPLACE) && is_ecommerce_active();
    }

    private function validateAdTypePresence($field)
    {
        if ($this->ad?->adType) {
            return $this->ad->adType?->{$field};
        }

        return false;
    }

    private function canHiddenPriceField($get){
        if ($this->hasEnableOnlineShopping())  return false;

        if ($this->validateAdTypePresence('disable_price_type') || $get('price_type_id') != 1) {
            return true;
        }
    }
}
