<?php

namespace App\Filament\App\Resources;

use App\Filament\App\Resources\MySaleResource\Pages;
use App\Models\Page;
use App\Models\Reservation\Order;
use App\Models\Reservation\OrderStatusHistory;
use App\Models\Wallets\Wallet;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\SelectColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Builder;

class MySaleResource extends Resource
{
    protected static ?string $model = Order::class;

    public static function getNavigationGroup(): ?string
    {
        return __('messages.t_ecommerce_navigation');
    }

    public static function isDiscovered(): bool
    {
        return is_ecommerce_active() || isEnablePointSystem();
    }

    public static function getNavigationLabel(): string
    {
        return __('messages.t_my_sales');
    }

    public function getTitle(): string
    {
        return __('messages.t_my_sales');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(fn(Builder $query) => $query->where('vendor_id', auth()->id()))
            ->emptyStateIcon('/images/not-found.svg')
            ->columns(self::getColumns())
            ->filters([
                //
            ])
            ->actions([
                Action::make('order_accept')
                    ->action(function ($record) {
                        if ($record->order_type == RESERVATION_TYPE_POINT_VAULT) {
                            // Handle wallet and transaction for vendor
                            self::handleWalletTransaction($record->vendor_id, $record->points, $record->order_number, $record->id, 'Order received');

                            // Handle wallet and transaction for user
                            self::handleWalletTransaction($record->user_id, $record->points, $record->order_number, $record->id, 'Purchase Order', false);

                            // Update payment status and order history
                            $record->payment_status = 'completed';
                            $record->save();
                        }
                        $record->histories()->where('action', 'order_accepted')->first()?->update(['action_date' => now()]);
                    })
                    ->button()
                    ->visible(function ($record) {
                        return $record->histories()->whereNull('action_date')->where('action', 'order_accepted')->exists() &&
                            $record->histories()->whereNull('action_date')->where('action', 'order_rejected')->exists();
                    })
                    ->label(__('messages.t_my_sale_order_accept_button'))
                    ->requiresConfirmation()
                    ->modalHeading(__('messages.t_my_sale_confirm_order_acceptance_heading'))
                    ->modalDescription(__('messages.t_my_sale_confirm_order_acceptance_description'))
                    ->modalContent(function ($record) {
                        $page = Page::find(getPointSystemSetting('policy_page'));

                        return view('filament.modals.order-accept', ['record' => $record, 'slug' => $page->slug]);
                    })->modalButton(__('messages.t_my_sale_order_accept_button')),

                // Order reject action
                Action::make('order_reject')
                    ->action(function ($record) {
                        // Update rejection history and payment status
                        $record->histories()->where('action', 'order_rejected')->first()?->update(['action_date' => now()]);
                        if ($record->order_type == RESERVATION_TYPE_POINT_VAULT) {
                            $record->payment_status = 'refund';
                            $record->save();

                            // Handle wallet and transaction for user on refund
                            self::handleWalletTransaction($record->user_id, $record->points, $record->order_number, $record->id, 'Order was rejected. The points have been refunded to your wallet.', true);
                        }
                    })
                    ->requiresConfirmation()
                    ->button()
                    ->color('danger')
                    ->modalHeading(__('messages.t_my_sale_confirm_order_reject_heading'))
                    ->modalDescription(__('messages.t_my_sale_confirm_order_reject_description'))
                    ->label(__('messages.t_my_sale_order_reject_button'))
                    ->visible(function ($record) {
                        return $record->histories()->whereNull('action_date')->where('action', 'order_accepted')->exists() &&
                            $record->histories()->whereNull('action_date')->where('action', 'order_rejected')->exists();
                    }),

                // View action
                Action::make('view')
                    ->icon('heroicon-o-eye')
                    ->label(__('messages.t_view_my_purchases'))
                    ->action(fn($record) => redirect()->route('reservation.view-purchases', $record->id))
                    ->openUrlInNewTab(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getColumns(): array
    {
        return [
            TextColumn::make('order_number')
                ->label(__('messages.t_my_sale_order_number'))
                ->sortable(),

            TextColumn::make('order_date')
                ->label(__('messages.t_my_sale_order_date'))
                ->sortable(),

            TextColumn::make('points')
                ->label(__('messages.t_my_sale_points'))
                ->visible(fn($livewire) => isEnablePointSystem())
                ->sortable(),

            TextColumn::make('total_amount')
                ->label(__('messages.t_my_sale_total_amount'))
                ->hidden(fn($livewire) => isEnablePointSystem())
                ->sortable(),

            TextColumn::make('discount_amount')
                ->label(__('messages.t_my_sale_discount_amount'))
                ->hidden(fn($livewire) => isEnablePointSystem())
                ->sortable(),

            TextColumn::make('subtotal_amount')
                ->label(__('messages.t_my_sale_subtotal_amount'))
                ->hidden(fn($livewire) => isEnablePointSystem())
                ->sortable(),

            TextColumn::make('payment_method')
                ->badge()
                ->hidden(fn($livewire) => isEnablePointSystem())
                ->color(fn(string $state): string => self::getPaymentMethodColor($state))
                ->label(__('messages.t_my_sale_payment_method'))
                ->sortable(),

            TextColumn::make('payment_status')
                ->badge()
                ->color(fn(string $state): string => $state === 'completed' ? 'success' : ($state === 'refund' ? 'danger' : 'info'))
                ->formatStateUsing(fn(string $state): string => Str::title($state))
                ->label(fn($livewire) => (isEnablePointSystem())
                    ? __('messages.t_my_sale_point_earned')
                    : __('messages.t_my_sale_payment_status'))
                ->sortable(),

            TextColumn::make('transaction_id')
                ->label(__('messages.t_my_sale_transaction_id'))
                ->hidden(fn($livewire) => isEnablePointSystem())
                ->sortable(),

            TextColumn::make('status')
                ->badge()
                ->default(fn($record) => self::getLatestOrderStatus($record))
                ->color(fn(string $state): string => self::getOrderStatusColor($state))
                ->formatStateUsing(fn(string $state): string => str_replace('_', ' ', Str::title($state)))
                ->label(__('messages.t_my_sale_status'))
                ->sortable(),

            SelectColumn::make('order_status')
                ->disabled(fn($record) => $record ? !$record->histories()->whereNotNull('action_date')->where('action', 'order_accepted')->exists() : false)
                ->options(fn($record) => self::getOrderStatusOptionsForSelect($record))
                ->updateStateUsing(fn($record, $state) => self::updateAdStatus($record, $state))
                ->disableOptionWhen(fn(string $value, $record): bool => self::isStatusOptionDisabled($value, $record))
                ->label(__('messages.t_my_sale_change_status_action'))

        ];
    }

    public static function getPaymentMethodColor(string $state): string
    {
        return match ($state) {
            'stripe', 'paypal', 'flutterwave', 'paymongo', 'paystack', 'razorpay' => 'success',
            'offline' => 'info',
            default => 'secondary',
        };
    }

    public static function getLatestOrderStatus($record): ?string
    {
        return $record->histories()
            ->whereNotNull('action_date')
            ->orderBy('updated_at', 'desc')
            ->first()?->action;
    }

    public static function getOrderStatusColor(string $state): string
    {
        return match ($state) {
            'order_requested', 'order_accepted', 'order_received' => 'info',
            'order_processed', 'order_shipped' => 'warning',
            'order_delivered' => 'success',
            'order_cancelled', 'order_rejected' => 'danger',
            default => 'secondary',
        };
    }

    public static function getOrderStatusOptions($record): array
    {
        $status = OrderStatusHistory::where('order_id', $record->id)->pluck('action')->toArray();
        $action = OrderStatusHistory::where('order_id', $record->id)
            ->whereNull('action_date')->first()?->action;

        $array = [
            'order_requested' =>  __('messages.t_order_request'),
            'order_accepted' =>  __('messages.t_order_accepted'),
            'order_processed' =>  __('messages.t_order_processed'),
            'order_shipped' =>  __('messages.t_order_shipped'),
            'order_delivered' => __('messages.t_order_delivered'),
            'order_received' =>  __('messages.t_order_received'),
            'order_cancelled' =>  __('messages.t_order_cancelled'),
            'order_rejected' => __('messages.t_order_rejected'),
        ];

        $returnArray = [];

        foreach ($status as $value) {
            $returnArray[$value] = $array[$value];

            if ($action == $value) break;
        }

        return $returnArray;
    }

    public static function getOrderStatusOptionsForSelect($record): array
    {
        $status = OrderStatusHistory::where('order_id', $record->id)
            ->whereNotIn('action', ['order_received', 'order_rejected'])
            ->pluck('action')
            ->toArray();

        $action = OrderStatusHistory::where('order_id', $record->id)
            ->whereNull('action_date')->first()?->action;

        $array = [
            'order_requested' =>  __('messages.t_order_request'),
            'order_accepted' =>  __('messages.t_order_accepted'),
            'order_processed' =>  __('messages.t_order_processed'),
            'order_shipped' =>  __('messages.t_order_shipped'),
            'order_delivered' => __('messages.t_order_delivered'),
            'order_received' =>  __('messages.t_order_received'),
            'order_cancelled' =>  __('messages.t_order_cancelled')
        ];

        $returnArray = [];

        foreach ($status as $value) {
            $returnArray[$value] = $array[$value];

            if ($action == $value) break;
        }

        return $returnArray;
    }

    public static function isStatusOptionDisabled(string $value, $record): bool
    {
        $completedActions = OrderStatusHistory::where('order_id', $record->id)
            ->whereNotNull('action_date')
            ->pluck('action')
            ->toArray();

        return in_array($value, $completedActions);
    }

    public static function updateAdStatus($record, string $state)
    {
        $record->histories()->where('action', $state)->first()?->update([
            'action_date' => now()
        ]);

        Notification::make()
            ->title(__('messages.t_updated_successfully'))
            ->success()
            ->send();
    }

    public static function handleWalletTransaction($userId, $points, $orderNumber, $orderId, $transactionType, $isAdded = true)
    {
        // Fetch or create the user's wallet
        $wallet = Wallet::firstOrCreate(
            ['user_id' => $userId],  // Query condition
            ['points' => 0] // Default balance for new wallet
        );

        // Increment wallet balance
        if ($isAdded) {
            $wallet->increment('points', $points);
        }

        // Create a transaction record for the wallet
        $wallet->transactions()->create([
            'user_id' => $userId,
            'points' => $points,
            'transaction_reference' => $orderNumber,
            'transaction_type' => $transactionType,
            'is_added' => $isAdded,
            'status' => 'completed',
            'payable_type' => Order::class,  // Polymorphic model type
            'payable_id' => $orderId,        // Polymorphic model id
        ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListMySales::route('/'),
            'create' => Pages\CreateMySale::route('/create'),
            'edit' => Pages\EditMySale::route('/{record}/edit'),
        ];
    }
}
