<?php

namespace Adfox\VehicleRentalMarketplace\Livewire;

use Adfox\VehicleRentalMarketplace\Models\BookingAddress;
use Adfox\VehicleRentalMarketplace\Models\TemproveryBookingData;
use Adfox\VehicleRentalMarketplace\Models\VehicleCarBooking;
use App\Models\City;
use App\Models\Country;
use App\Models\State;
use App\Settings\PaymentSettings;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Select;
use Illuminate\Support\Str;
use Livewire\Component;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Form;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Get;
use Filament\Notifications\Notification;

class VehicleSummary extends Component implements HasForms
{
    use InteractsWithForms;

    public $summeryDetails;
    public $deliveryAddress;
    public ?array $data = [];
    public $locations;
    public $tax = 0;

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Group::make()->schema([
                    TextInput::make('name')
                        ->required(),
                    TextInput::make('phone_number')
                        ->numeric()
                        ->required(),
                    TextInput::make('house_number')
                        ->required(),
                    TextInput::make('address')
                        ->required(),
                    Select::make('country_id')
                        ->label('Country')
                        ->options(Country::pluck('name', 'id')->toArray())
                        ->live()
                        ->afterStateUpdated(fn(callable $set) => $set('state_id', null))
                        ->required(),
                    Select::make('state_id')
                        ->label('State')
                        ->options(function (Get $get) {
                            $countryId = $get('country_id');
                            if (!$countryId) {
                                return [];
                            }
                            return State::where('country_id', $countryId)->pluck('name', 'id')->toArray();
                        })
                        ->live()
                        ->hidden(fn(Get $get): bool => !$get('country_id'))
                        ->afterStateUpdated(fn(callable $set) => $set('city_id', null))
                        ->required(),
                    Select::make('city_id')
                        ->label('City')
                        ->options(function (Get $get) {
                            $stateId = $get('state_id');
                            if (!$stateId) {
                                return [];
                            }
                            return City::where('state_id', $stateId)->pluck('name', 'id')->toArray();
                        })
                        ->hidden(fn(Get $get): bool => !$get('state_id'))
                        ->required(),
                    TextInput::make('postal_code')
                        ->required(),

                ])->columns(2)
            ])
            ->statePath('data');
    }

    public function create(): void
    {
        $data = $this->form->getState('data');
        BookingAddress::create([
            'name' => $data['name'],
            'address' => $data['address'],
            'city_id' => $data['city_id'],
            'state_id' => $data['state_id'],
            'country_id' => $data['country_id'],
            'postal_code' => $data['postal_code'],
            'user_id' => auth()->id(),
            'house_number' => $data['house_number'],
            'phone_number' => $data['phone_number'],
        ]);
        $this->js('location.reload();');
    }

    protected function initializeDeliveryAddress()
    {
        $this->locations = auth()->user()->bookinglocations()->get();

        if ($this->locations->isEmpty()) {
            session()->put('delivery-address', null);
        } else if ($this->locations->isNotEmpty() && !session()->has('delivery-address')) {
            session()->put('delivery-address', $this->locations[0]);
        }

        $this->deliveryAddress = session()->get('delivery-address', null);
    }

    public function selectAddress($addressId)
    {
        $location = BookingAddress::find($addressId);
        session()->put('delivery-address',  $location);

        $this->deliveryAddress = session()->get('delivery-address',  null);
    }

    public function mount($ad_id)
    {
        $this->summeryDetails = TemproveryBookingData::where('ad_id', $ad_id)->where('user_id', auth()->id())->latest()->first();
        $this->initializeDeliveryAddress();
        $paymentSettings = app(PaymentSettings::class);

        if ($paymentSettings->enable_tax) {
            if ($paymentSettings->tax_type === 'percentage') {
                // Calculate tax as a percentage of the subtotal
                $this->tax = (($this->summeryDetails->ads->price * $this->summeryDetails->days) * $paymentSettings->tax_rate) / 100;
            } else if ($paymentSettings->tax_type === 'fixed') {
                // Apply a fixed tax rate
                $this->tax = $paymentSettings->tax_rate;
            }
        } else {
            // No tax applied
            $this->tax = 0;
        }

    }

    public function bookingConfirm()
    {
        if (isset($this->deliveryAddress)) {
            $bookingId = 'BI-' . Str::random(10) . "-" . (VehicleCarBooking::count() + 1);
            $total = $this->summeryDetails->ads->price * $this->summeryDetails->days + $this->tax;
            $bookingId =  VehicleCarBooking::create([
                'ad_id' => $this->summeryDetails->ad_id,
                'user_id' => auth()->id(),
                'start_date' => $this->summeryDetails->start_date,
                'end_date' => $this->summeryDetails->end_date,
                'days' => $this->summeryDetails->days,
                'status' => 'booked',
                'payment_type' => 'offline',
                'booking_id' => $bookingId,
                'total' => $total,
                'booking_address_id' => $this->deliveryAddress->id
            ]);
            return redirect()->route('sucess', ['id' => $bookingId->id]);
        } else {
            Notification::make()
                ->title(__('choose your address'))
                ->danger()
                ->send();
            return;
        }
    }

    public function render()
    {
        return view('vehicle-rental-marketplace::livewire.vehicle-summary');
    }
}
