<?php

namespace Adfox\VehicleRentalMarketplace\Livewire;

use App\Models\Ad;
use Adfox\VehicleRentalMarketplace\Models\VehicleFeature;
use Adfox\VehicleRentalMarketplace\Models\VehicleFuelType;
use Adfox\VehicleRentalMarketplace\Models\VehicleMake;
use Adfox\VehicleRentalMarketplace\Models\VehicleTransmission;
use Adfox\VehicleRentalMarketplace\Models\VehicleRental;
use Adfox\VehicleRentalMarketplace\Models\VehicleRentalAd;
use App\Livewire\Ad\PostAd\AdDetail;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Radio;
use Filament\Forms\Components\CheckboxList;
use Filament\Forms\Form;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Schema;
use App\Foundation\AdBase\BaseAdDetail;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Get;
use Livewire\Component;
use Filament\Forms\Components\MarkdownEditor;
use Filament\Forms\Components\ToggleButtons;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Set;

//
class VehicleAdDetail extends BaseAdDetail
{
    public ?array $businessData = [];

    public $make_id;
    public $availability_window;
    public $min_trip_length;
    public $max_trip_length;
    public $transmission_id;
    public $fuel_type_id;
    public $mileage;
    public $features;

    protected function fillAdDetails()
    {
        $this->title = $this->ad->title;
        $this->description = $this->ad->description;
        $this->price = $this->ad->price;
        $this->type = $this->ad->type;
        $this->for_sale_by = $this->ad->for_sale_by;
        $this->tags = $this->ad->tags ? $this->ad->tags : [];
        $this->price_type_id = $this->ad->price_type_id;
        $this->display_phone = $this->ad->display_phone;
        $this->phone_number = $this->ad->phone_number;
        $this->condition_id = $this->ad->condition_id;
        $this->price_suffix = $this->ad->price_suffix;
        $this->offer_price = $this->ad->offer_price;
        $this->make_id = $this->ad->make_id;
        $this->availability_window = $this->ad->availability_window;
        $this->min_trip_length = $this->ad->min_trip_length;
        $this->max_trip_length = $this->ad->max_trip_length;
        $this->transmission_id = $this->ad->transmission_id;
        $this->fuel_type_id = $this->ad->fuel_type_id;
        $this->mileage = $this->ad->mileage;
        $vehicle = VehicleRentalAd::find($this->id);
        $this->features = $vehicle->features->pluck('id') ?? [];

        if ($this->ad->category_id) {
            $this->category_id = $this->ad->category_id;
            if ($this->ad->category) {
                $this->parent_category = $this->ad->category->parent_id;
                $this->showMainCategories = false;
                $this->disable_condition = $this->ad->category->disable_condition;
                $this->disable_price_type = $this->ad->category->disable_price_type;
            }
        }
    }


    public function checkRequiredFieldsFilled()
    {
        $isFilled = false;
        foreach ($this->requiredFields as $field) {
            if (trim($this->$field) !== '' && (!empty(trim($this->$field))) && (!is_null(trim($this->$field)))) {
                $isFilled = true;
            } else {
                $isFilled = false;
                break;
            }
        }

        if (isset($this->display_phone) && $this->display_phone && (!empty($this->phone_number))) {
            $isFilled = $isFilled ? true : false;
        }

        $this->dispatch('required-fields-filled', isFilled: $isFilled);
    }

    public function getRequiredFieldsProperty()
    {
        $requiredFields = [];
        $rules = $this->getRules();
        // dd($rules);
        foreach ($rules as $field => $rule) {
            if (is_array($rule) && in_array('required', $rule)) {
                $requiredFields[] = $field;
            } elseif ($rule == 'required') {
                $requiredFields[] = $field;
            }
        }
        return $requiredFields;
    }

    public function businessSpecificForm(): Form
    {
        // Regular ads don't have business-specific fields
        return $this->makeForm()->schema([

            // ...self::getVehicleDetailForm(),
        ]);
    }

    public function getVehicleDetailForm()
    {
        return [
            Select::make('make_id')
                ->options(VehicleMake::all()->pluck('name', 'id'))
                ->label(__('messages.t_make'))
                ->live(onBlur: true)
                ->required(),
            Select::make('availability_window')
                ->live(onBlur: true)
                ->label(__('messages.t_availability_window'))
                ->options([
                    'all_future_dates' => 'All future dates',
                    '3_months_into_the_feature' => '3 months into the feature',
                    '6_months_into_the_feature' => '6 months into the feature',
                    '9_months_into_the_feature' => '9 months into the feature',
                ])
                ->required(),
            Grid::make(['default' => 2])
                ->schema([
                    TextInput::make('min_trip_length')
                        ->required()
                        ->label(__('messages.t_min_trip_length'))
                        ->numeric()
                        ->prefix(__('messages.t_days'))
                        ->live(onBlur: true)
                        ->lte(function(Get $get){
                           if($get('max_trip_length')){
                            return 'max_trip_length';
                           }
                        })
                        ->minValue(1),
                    TextInput::make('max_trip_length')
                        ->required()
                        ->gte(function(Get $get){
                            if($get('min_trip_length')){
                             return 'min_trip_length';
                            }
                         })
                        ->prefix(__('messages.t_days'))
                        ->label(__('messages.t_max_trip_length'))
                        ->numeric()
                        ->live(onBlur: true),
                ]),
            Select::make('transmission_id')
                ->required()
                ->columns(4)
                ->live(onBlur: true)
                ->label(__('messages.t_transmission'))
                ->options(VehicleTransmission::all()->pluck('name', 'id')),
            Radio::make('fuel_type_id')
                ->required()
                ->columns(4)
                ->live(onBlur: true)
                ->label(__('messages.t_fuel_type'))
                ->options(VehicleFuelType::all()->pluck('name', 'id')),
            TextInput::make('mileage')
                ->required()
                ->label(__('messages.t_mileage'))
                ->numeric()
                ->live(onBlur: true)
                ->prefix(__('messages.t_mileage_prefix'))
                ->minValue(0),
            CheckboxList::make('features')
                ->columns(3)
                ->live(onBlur: true)
                ->label(__('messages.t_vehicle_features'))
                ->options(VehicleFeature::all()->pluck('name', 'id')),
        ];
    }

    public function detailForm(): Form
    {
        return $this->makeForm()->schema([
            ToggleButtons::make('for_sale_by')
                ->label(__('messages.t_for_sale_by'))
                ->live()
                ->grouped()
                ->options([
                    'owner' => __('messages.t_owner_for_sale'),
                    'business' => __('messages.t_business_for_sale'),
                ]),
            MarkdownEditor::make('description')
                ->disableToolbarButtons([
                    'attachFiles'
                ])
                ->label(__('messages.t_description'))
                ->live(onBlur: true)
                ->minLength(20)
                ->required(),
            Select::make('price_suffix')
                ->default('days')
                ->label(__('messages.t_price'))
                ->required()
                ->default('day')
                ->options(['day' => __('messages.t_per_day')]),
            TextInput::make('price')
                ->label(__('messages.t_price_your_ad'))
                ->hiddenLabel()
                ->helperText(__('messages.t_set_fair_price'))
                ->required()
                ->validationAttribute(__('messages.t_set_fair_price'))
                ->numeric()
                ->minValue(1)
                ->placeholder(__('messages.t_price_your_ad'))
                ->prefix(config('app.currency_symbol'))
                ->live(onBlur: true),
                ...self::getVehicleDetailForm()
        ]);
    }

    public function updated($name, $value)
    {
        $this->checkRequiredFieldsFilled();
        $userId = auth()->id();
        $this->validateOnly($name);
        if (!$this->id) {
            if (!$userId) {
                abort(403, 'Unauthorized action.');
            }
            $ad = Ad::create([$name => $value, 'user_id' => $userId]);
            $this->id = $ad->id;
            $this->updateAdSlug($ad, $value);
            $this->loadAdDetails($this->id);
        } else {
            $ad = Ad::find($this->id);

            if (!$ad || $ad->user_id != $userId) {
                abort(403, 'Unauthorized action.');
            }
            // Update the "tags" property if it's an array element update
            if (str_starts_with($name, 'tags.')) {
                $index = explode('.', $name)[1];
                $tags = $this->tags;
                $tags[$index] = $value;
                $this->tags = $tags;
                $ad->update(['tags' => json_encode($tags)]);
            } else {
                // Update other fields
                $adData = [$name => $value];
                $ad->update($adData);
                if ($name == 'title') {
                    // If the title is updated, update the slug too
                    $this->updateAdSlug($ad->fresh(), $value);
                }
                if ($name == 'category_id') {
                    $this->disable_condition = $ad->category->disable_condition;
                    $this->disable_price_type = $ad->category->disable_price_type;

                    // If the sub category is updated, update the location if it has default location
                    $this->updateLocation($ad, $value);
                }
            }
            $vehicle = VehicleRentalAd::find($this->id);
            if ($name == 'availability_window') {
                $availabilityDates = $this->getStartAndEndDate($value);
                $vehicle->update($availabilityDates);
            }

            if (str_starts_with($name, 'features.')) {
                $vehicle->features()->sync($this->features);
            }
        }
        $this->dispatch('ad-created', id: $ad->id);
    }

    public function getStartAndEndDate($availabilityWindow)
    {
        switch ($availabilityWindow) {
            case 'all_future_dates':
                return ['start_date' => Carbon::now(), 'end_date' => null];
            case '3_months_into_the_feature':
                return ['start_date' => Carbon::now(), 'end_date' => Carbon::now()->addMonths(3)];
            case '6_months_into_the_feature':
                return ['start_date' => Carbon::now(), 'end_date' => Carbon::now()->addMonths(6)];
            case '9_months_into_the_feature':
                return ['start_date' => Carbon::now(), 'end_date' => Carbon::now()->addMonths(9)];
            default:
                return ['start_date' => Carbon::now(), 'end_date' => null];
        }
    }

    public function save()
    {
        $this->validate();

        $adData = $this->only([
            'title',
            'category_id',
            'description',
            'price',
            'price_type_id',
            'condition_id',
            'display_phone',
            'phone_number',
            'for_sale_by',
            'tags',
            'price_suffix',
            'offer_price'
        ]);

        if ($this->id) {
            $ad = Ad::find($this->id);
            $ad->update($adData);
        } else {
            $ad = Ad::create(array_merge($adData, ['user_id' => auth()->id()]));
        }

        $this->updateAdSlug($ad, $this->title);

        $this->dispatch('ad-saved', id: $ad->id);
        return $this->redirect('/ads');
    }
}
