<?php

namespace Adfox\Paystack\Http\Controllers\Callback;

use App\Http\Controllers\Controller;
use App\Models\{WebhookUpgrade, WebhookPackage};
use App\Settings\PaystackSettings;
use Illuminate\Http\Request;
use App\Settings\FlutterwaveSettings;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use App\Traits\HandlesPromotionApplication;
use App\Traits\HandlesPackageApplication;
use Illuminate\Contracts\Session\Session;

class PaystackController extends Controller
{
    private $paystackSettings;
    use HandlesPackageApplication;
    use HandlesPromotionApplication;

    public function __construct(PaystackSettings $paystackSettings)
    {
        $this->paystackSettings = $paystackSettings;
    }
    /**
     * Handle the callback from the Stripe payment gateway.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function callback(Request $request)
    {
        //Get reference id
        $reference = $request->get('reference');

        // Check webhook event
        if ($reference) {
            $response = $this->verifyPayment($reference);

            if ($response['success'] === true) {
                // Get order id
                $order_id = $response['response']->metadata->order_id;
                // Check If Package Management Callback
                if (Str::startsWith($order_id, "PKG-")) {
                    $orderData = WebhookPackage::where('payment_id', $order_id)
                        ->where('payment_method', 'paystack')
                        ->where('status', 'pending')
                        ->firstOrFail();

                    $routeParameters = $this->applyPackages($orderData);
                    return redirect()->route('package-success', $routeParameters);

                } else {
                    $orderData = WebhookUpgrade::where('payment_id', $order_id)
                        ->where('payment_method', 'paystack')
                        ->where('status', 'pending')
                        ->firstOrFail();
                    $routeParameters = $this->applyPromotionsWithRedirect($orderData);
                    return redirect()->route('success-upgrade', $routeParameters);
                }
            }
        }
        return redirect('/');
    }


    /**
     * Verify if payment succeeded
     *
     * @param string $id
     * @return array
     */
    private function verifyPayment($reference)
    {
        try {
            $headers = [
                "Cache-Control: no-cache",
            ];

            //Url information
            $parameters=[
                'endpoint' => 'https://api.paystack.co/transaction/verify',
                'reference' => $reference,
            ];
            
            //Make the http request to verify the transaction
            $response = Http::withToken($this->paystackSettings->secret_key)
                ->withUrlParameters($parameters)
                ->withHeaders($headers)
                ->get('{+endpoint}/{reference}');

            //Handle successful response
            if ($response->successful()) {
                $data = $response->object();
                // Check if payment succeeded
                if ($data->data->status === 'success') {
                    // Done
                    return [
                        'success' => true,
                        'response' => $data->data
                    ];
                }
            } else {
                return [
                    'success' => false,
                    'message' => __('messages.t_error_payment_failed')
                ];
            }
        } catch (\Throwable $th) {

            // Error
            return [
                'success' => false,
                'message' => __('messages.t_toast_something_went_wrong')
            ];

        }
    }

}

