<?php

namespace Adfox\LiveChat\Http\Controllers\Chat;

use App\Models\User;
use App\Utils\Chat\ChatApi;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use App\Models\Message;
use Illuminate\Support\Facades\Cache;
use App\Models\Conversation;
use Illuminate\Support\Facades\Response;
use App\Notifications\User\Everyone\NewMessage;
use App\Settings\GeneralSettings;
use App\Settings\LiveChatSettings;
use App\Settings\SEOSettings;
use Artesaos\SEOTools\Traits\SEOTools as SEOToolsTrait;
use Illuminate\Support\Facades\Log;

class MessagesController extends Controller
{
    use SEOToolsTrait;

    protected $chat;
    protected $perPage = 30;


    /**
     * Set chat api
     */
    public function __construct() {
        $this->chat = new ChatApi();
    }


    /**
     * Authenticate the connection for pusher
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function pusherAuth(Request $request)
    {
        return $this->chat->pusherAuth(
            $request->user(),
            auth()->user(),
            $request->get('channel_name'),
            $request->get('socket_id')
        );
    }


    /**
     * eturning the view of the app with the required data
     *
     * @param string $id
     * @return \Illuminate\Contracts\View\View
     */
    public function index( $id = null)
    {
        // Get user
        $conversation  = Conversation::find($id);

        $generalSettings = app(GeneralSettings::class);
        $seoSettings = app(SEOSettings::class);


        $separator = $generalSettings->separator ?? '-';
        $siteName = $generalSettings->site_name ?? app_name();

        $title = __('messages.t_seo_my_messages_page_title') . " $separator " . $siteName;
        $description = $seoSettings->meta_description;
        $this->seo()->setTitle($title);
        $this->seo()->setDescription($description);

        // Return view
        return view('Chatify::pages.app', [
            'id'             => $conversation ? $conversation->id : 0,
            'type'           => 'user',
            'messengerColor' => '#000000',
            'dark_mode'      => 'light',
        ]);
    }


    /**
     * Fetch user by id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function idFetchData(Request $request)
    {
        $conversationId = $request->get('id');
        $userId = auth()->id();

       // Fetch the conversation and ensure the authenticated user is a participant
        $conversation = Conversation::with(['buyer', 'seller', 'ad'])
        ->where('id', $conversationId)
        ->where(function ($query) use ($userId) {
            $query->where('buyer_id', $userId)
                ->orWhere('seller_id', $userId);
        })
        ->first();

        // Initialize variables
        $favorite = false;
        $fetch = null;
        $userAvatar = null;

        // Check if conversation exists
        if ($conversation) {
            $userId = auth()->id();

            // Fetch the other user's data
            $otherUser = $conversation->buyer_id == $userId ? $conversation->seller : $conversation->buyer;

            if ($otherUser) {
                $fetch = [
                    'id' => $otherUser->id,
                    'name' => $otherUser->name,
                    'slug' => $otherUser->slug,
                ];
                $adAvatar = $conversation->ad->primary_image;
                // Set avatar
                $userAvatar = $otherUser->avatar; // Ensure you have a method to get the user's avatar
            }
        }

        // Send the response
        return Response::json([
            'favorite'    => $favorite,
            'fetch'       => $fetch,
            'user_avatar' => $userAvatar,
            'ad_avatar' => $adAvatar,
        ]);
    }



    /**
     * This method to make a links for the attachments
     * to be downloadable.
     *
     * @param string $fileName
     * @return \Symfony\Component\HttpFoundation\StreamedResponse|void
     */
    public function download($fileName)
    {
        // Get file path
        $path = config('chatify.attachments.folder') . '/' . $fileName;

        // Check if file exists
        if ($this->chat->storage()->exists($path)) {
            return $this->chat->storage()->download($path);
        }

        // File not found
        return abort(404, __('messages.t_sorry_file_chat_does_not_exist'));
    }


    /**
     * Send a message to database
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function send(Request $request)
    {
        // default variables
        $error = (object)[
            'status'  => 0,
            'message' => null
        ];

        // Set default attachement values
        $attachment       = null;
        $attachment_title = null;

        // if there is attachment [file]

        if ($request->hasFile('audio')) {
            $audioFile = $request->file('audio');
            $audioFileName = uniqid() . '.' . $audioFile->getClientOriginalExtension();

            $audioFile->storeAs(config('chatify.attachments.folder'), $audioFileName, config('chatify.storage_disk_name'));

            // Use the stored file name as attachment for the message
            $attachment = [
                'new_name' => $audioFileName,
                'old_name' => $audioFile->getClientOriginalName(),
                'size' => $audioFile->getSize(),
                'extension' => $audioFile->getClientOriginalExtension()
            ];
        } elseif ($request->hasFile('file') && app(LiveChatSettings::class)->enable_uploading_attachments) {

            // allowed extensions
            $allowed_images = $this->chat->getAllowedImages();
            $allowed_files  = $this->chat->getAllowedFiles();
            $allowed        = array_merge($allowed_images, $allowed_files);

            // Get file
            $file           = $request->file('file');

            // Check file size
            if ($file->getSize() < $this->chat->getMaxUploadSize()) {

                // Check  extension
                if (in_array(strtolower($file->extension()), $allowed)) {

                    // Get attachment name
                    $attachment_title = $file->getClientOriginalName();

                    // Set new name for this attachment
                    $attachment       = Str::uuid() . "." . $file->extension();

                    // Upload attachment and store the new name
                    $file->storeAs(config('chatify.attachments.folder'), $attachment, config('chatify.storage_disk_name'));

                    $attachment = [
                        'new_name' => $attachment,
                        'old_name' => htmlentities(trim(clean($attachment_title)), ENT_QUOTES, 'UTF-8'),
                        'size' => $file->getSize(),
                        'extension' => $file->extension()
                    ];

                } else {

                    // Error
                    $error->status  = 1;
                    $error->message = __('messages.t_selected_file_extension_is_not_allowed');

                }



            } else {

                // Error
                $error->status  = 1;
                $error->message = __('messages.t_selected_file_size_big');

            }

        } else {

            // Get message
            $get_message = clean($request->get('message'));

            // Check if message is empty or not set
            if (!$get_message) {

                // Error
                $error->status  = true;
                $error->message = __('messages.t_enter_your_message');

            }

        }

        // Check if there is no error
        if (!$error->status) {
            // Fetch the conversation
            $conversation = Conversation::find($request->get('id'));
            if (!$conversation) {
                return response()->json([
                    'status' => '404',
                    'error' => ['status' => 1, 'message' => 'Conversation not found.']
                ]);
            }

            // Determine the receiver ID based on the conversation
            $receiverId = $conversation->buyer_id == auth()->id() ? $conversation->seller_id : $conversation->buyer_id;

            // Generate message id
            $message_id = mt_rand(9, 999999999) + time();

            $attachmentJson = $attachment ? json_encode($attachment) : null;

            // Save message
            $this->chat->newMessage([
                'id' => $message_id,
                'conversation_id' => $conversation->id,
                'sender_id' => auth()->id(),
                'receiver_id' => $receiverId, // Add receiver_id
                'content' => $request->get('message') ? clean($request->get('message')) : null,
                'attachment' =>  $attachmentJson,
            ]);

            // fetch message to send it with the response
            $messageData = $this->chat->fetchMessage($message_id);

            // Send to user using pusher
            $this->chat->push("private-chat.".$receiverId, 'messaging', [
                'from_id' => $request->get('id'),
                'to_id' => $receiverId,
                'message' => $this->chat->messageCard($messageData, 'default')
            ]);

            // [Additional code for notifications and response]
        }

        // Send the response
        return Response::json([
            'status'  => '200',
            'error'   => $error,
            'message' => $this->chat->messageCard(@$messageData),
            'tempID'  => $request->get('temporaryMsgId'),
        ]);
    }


    /**
     * fetch [user/group] messages from database
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function fetch(Request $request)
    {
        // Fetch messages for the given conversation
        $messages = $this->chat->fetchMessagesQuery($request->get('id'))
                            ->latest()
                            ->paginate($request->per_page ?? $this->perPage);

        $totalMessages = $messages->total();
        $lastPage      = $messages->lastPage();

        // Prepare the response
        $response = [
            'total'           => $totalMessages,
            'last_page'       => $lastPage,
            'last_message_id' => collect($messages->items())->last()->id ?? null,
            'messages'        => '',
        ];

        // Handle case with no messages
        if ($totalMessages < 1) {
            $response['messages'] = '<p class="message-hint center-el"><span>' . __('messages.t_type_something_to_start_messaging') . '</span></p>';
            return Response::json($response);
        }

        // Build messages HTML
        $allMessages = '';
        foreach ($messages->reverse() as $index => $message) {
            $allMessages .= $this->chat->messageCard($this->chat->fetchMessage($message->id, $index));
        }

        // Update and return response
        $response['messages'] = $allMessages;
        return Response::json($response);
    }



    /**
     * Mark messages as seen
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function seen(Request $request)
    {
        // Make message as seen
        $seen = $this->chat->makeSeen($request->get('id'));

        // Send the response
        return response()->json([
            'status' => $seen
        ], 200);
    }

    /**
     * Get contacts list
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getContacts(Request $request)
    {
        $userId = auth()->id();

        // Fetch conversations for the authenticated user where they haven't deleted it
        $conversations = Conversation::with(['ad', 'buyer', 'seller', 'messages' => function($query) {
            $query->latest('updated_at')->take(1);
        }])
        ->where(function ($query) use ($userId) {
            $query->where('buyer_id', $userId)
                ->whereNull('deleted_by_buyer_at'); // Ensure the conversation is not deleted by the buyer
        })
        ->orWhere(function ($query) use ($userId) {
            $query->where('seller_id', $userId)
                ->whereNull('deleted_by_seller_at'); // Ensure the conversation is not deleted by the seller
        })
        ->orderByDesc(
            Message::select('updated_at')
                ->whereColumn('conversations.id', 'messages.conversation_id')
                ->orderBy('updated_at', 'desc')
                ->limit(1)
        )
        ->paginate($request->per_page ?? $this->perPage);




         // Check if list not empty
         if (count($conversations) > 0) {

             $formattedContacts = '';

             foreach ($conversations as $conversation) {
                $otherParty = $conversation->buyer_id === $userId ? $conversation->seller : $conversation->buyer;
                $contact = [
                    'title' => $conversation->ad->title,
                    'name'=> $otherParty->name,
                    'conversation_id' => $conversation->id,
                    'latest_message' => $conversation->messages->first(),
                    'ad_image' => $conversation->ad->primary_image,
                    'ad_title' => $conversation->ad->title,
                    'conversation_ad_id' => $conversation->ad_id,
                    'active_status' => false
                ];
                $formattedContacts .= $this->chat->getContactItem($contact, $conversation->id);
            }


         } else {
             // Contacts list empty
             $formattedContacts = '<p class="message-hint center-el w-full"><span>' . __('messages.t_ur_contact_list_is_empty') . '</span></p>';

         }

        // Send response
        return response()->json([
            'contacts'  => mb_convert_encoding($formattedContacts, 'UTF-8', 'UTF-8'),
            'total'     => $conversations->total() ?? 0,
            'last_page' => $conversations->lastPage() ?? 1,
        ], 200);
    }



    /**
     * Update user's list item data
     *
     * @param Request $request
     * @return JsonResponse
     */


     public function updateContactItem(Request $request)
    {
        // Fetch the conversation
        $conversation = Conversation::with(['ad', 'messages' => function($query) {
                            $query->latest('updated_at')->take(1);
                        }])
                        ->find($request->get('user_id'));

        // Check if conversation does not exist
        if (!$conversation) {
            return Response::json([
                'message' => __('messages.t_conversation_not_found'),
            ], 404);
        }

        // Determine the other party in the conversation
        $userId = auth()->id();
        $otherUserId = $conversation->buyer_id == $userId ? $conversation->seller_id : $conversation->buyer_id;

        // Get the other user's data
        $otherUser = User::find($otherUserId);

        // Check if the other user does not exist
        if (!$otherUser) {
            return Response::json([
                'message' => __('messages.t_user_not_found'),
            ], 401);
        }

        // Set contact with conversation details
        $contact = [
            'name' => $otherUser->name,
            'conversation_id' => $conversation->id,
            'latest_message' => $conversation->messages->first(),
            'ad_image' => $conversation->ad->primary_image,
            'ad_title' => $conversation->ad->title,
            'conversation_ad_id' => $conversation->ad_id,
            'active_status' => $otherUser->active_status,
        ];

        // Create contact item
        $contactItem = $this->chat->getContactItem($contact, $conversation->id);

        // Send the response
        return Response::json([
            'contactItem' => $contactItem,
        ], 200);
    }


    /**
     * Search in contacts
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function search(Request $request)
    {
        // Set empty records variable
        $getRecords = null;

        // Get input value
        $input      = trim(filter_var($request->get('input')));

        // Search contacts
        $records    = Message::join('users',  function ($join) {
                                $join->on('messages.sender_id', '=', 'users.id')
                                    ->orOn('messages.receiver_id', '=', 'users.id');
                            })
                            ->where(function ($q) {
                                $q->where('messages.sender_id', auth()->id())
                                ->orWhere('messages.receiver_id', auth()->id());
                            })
                            ->where(function($query) use($input) {
                                $query->where('users.name', 'LIKE', "%{$input}%");
                            })
                            ->where('users.id','!=',auth()->id())
                            ->groupBy('users.id')
                            ->paginate($request->per_page ?? $this->perPage);

        // Loop through records
        foreach ($records->items() as $record) {
            // Set view
            $getRecords .= view('Chatify::layouts.listItem', [
                'get' => 'search_item',
                'type' => 'user',
                'user' => $record,
            ])->render();
        }

        // Check if no results found
        if($records->total() < 1){
            $getRecords = '<p class="message-hint center-el"><span>' . __('messages.t_no_results_found') . '</span></p>';
        }

        // Send the response
        return response()->json([
            'records'   => $getRecords,
            'total'     => $records->total(),
            'last_page' => $records->lastPage()
        ], 200);

    }


    /**
     * Get shared photos
     *
     * @param Request $request
     * @return JsonResponse|void
     */
    public function sharedPhotos(Request $request)
    {
        // Get shared photos
        $shared       = $this->chat->getSharedPhotos($request->get('user_id'));

        // Set empty variable
        $sharedPhotos = null;

        // Shared with its template
        for ($i = 0; $i < count($shared); $i++) {

            // Set view
            $sharedPhotos .= view('Chatify::layouts.listItem', [
                'get'   => 'sharedPhoto',
                'image' => $this->chat->getAttachmentUrl($shared[$i]),
            ])->render();

        }

        // Send the response
        return Response::json([
            'shared' => count($shared) > 0 ? $sharedPhotos : '<p class="message-hint"><span>' . __('messages.t_nothing_shared_yet') . '</span></p>',
            'empty'  => count($shared)
        ], 200);
    }


    /**
     * Delete conversation
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function deleteConversation(Request $request)
    {
        // Delete conversation
        $delete = $this->chat->deleteConversation($request->get('id'));

        // send the response
        return Response::json([
            'deleted' => $delete ? 1 : 0,
        ], 200);
    }


    /**
     * Delete message
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function deleteMessage(Request $request)
    {
        // Delete message
        $delete = $this->chat->deleteMessage($request->get('id'));

        // send the response
        return Response::json([
            'deleted' => $delete ? 1 : 0,
        ], 200);
    }


    /**
     * Set user's active status
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function setActiveStatus(Request $request)
    {
        // Get user id
        $userId       = $request->get('user_id');

        // Get status
        $activeStatus = $request->get('status') > 0 ? 1 : 0;

        // Set status
        $status       = User::where('id', $userId)->update(['active_status' => $activeStatus]);

        // Send response
        return Response::json([
            'status' => $status,
        ], 200);
    }
}
