<?php

namespace Adfox\Feedback\Livewire;

use App\Models\Feedback;
use App\Models\FeedbackLike;
use App\Models\FeedbackReply;
use Livewire\Component;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Support\Enums\ActionSize;
use Filament\Support\Enums\MaxWidth;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\On;
use Livewire\Attributes\Reactive;

class FeedbackList extends Component implements HasForms, HasActions
{
    use InteractsWithActions;
    use InteractsWithForms;

    public $feedbackDetails;
    #[Reactive]
    public $activeTab;
    public $sellerId;

    public function mount()
    {
       $this->fetchFeedback();
    }

    #[On('fetch-feedback')]
    public function fetchFeedback() {
        if($this->sellerId) {
            $this->loadFeedbackDetails();
        } else {
            $this->loadFeedback();
        }
    }

    protected function loadFeedbackDetails()
    {
        $this->feedbackDetails = Feedback::with(['likes', 'replies', 'replies.user'])
            ->where('seller_id', $this->sellerId)
            ->get();
    }


    public function likeFeedback($feedbackId)
    {
        $existingLike = FeedbackLike::where('feedback_id', $feedbackId)
            ->where('user_id', Auth::id())
            ->first();

        if ($existingLike) {
            // If already liked, remove the like
            $existingLike->delete();

            // Optionally, you can add a notification for unlike action
            Notification::make()
                ->title(__('messages.t_feedback_unliked_successfully'))
                ->success()
                ->send();

        } else {
            // If not liked yet, add a new like
            FeedbackLike::create([
                'feedback_id' => $feedbackId,
                'user_id' => Auth::id(),
            ]);

            // Notification for like action
            Notification::make()
            ->title(__('messages.t_feedback_liked_successfully'))
            ->success()
            ->send();
        }
        $this->fetchFeedback();
    }




    public function loadFeedback()
    {
        if ($this->activeTab === 'sent') {
            $this->feedbackDetails = Feedback::where('buyer_id', Auth::id())->get();
        } else  {
            $this->feedbackDetails = Feedback::where('seller_id', Auth::id())->get();
        }
    }
     /**
     * Defines the action for replying to feedback.
     *
     * @param int $feedbackId The ID of the feedback being replied to.
     * @return Action
     */
    public function replyAction(): Action
    {
        return Action::make('replyAction')
            ->label(__('messages.t_reply'))
            ->form([
                Textarea::make('reply')
                    ->label(__('messages.t_your_reply'))
                    ->required()
                    ->maxLength(1000),
            ])
            ->link()
            ->extraAttributes(['class' => 'reply-action'])
            ->size(ActionSize::Large)
            ->modalWidth(MaxWidth::Medium)
            ->modalSubmitActionLabel(__('messages.t_send_reply'))
            ->action(function (array $data, array $arguments) {
                $this->handleReplySubmission($data, $arguments['feedbackId']);
            });
    }

    /**
     * Handles the submission of a reply to feedback.
     *
     * @param array $data Data from the reply form.
     * @param int $feedbackId The ID of the feedback being replied to.
     * @return void
     */
    protected function handleReplySubmission(array $data, $feedbackId)
    {
        try {
            FeedbackReply::create([
                'feedback_id' => $feedbackId,
                'user_id' => Auth::id(),
                'reply' => $data['reply'],
            ]);

            Notification::make()
                ->title(__('messages.t_reply_submitted_successfully'))
                ->success()
                ->send();
        } catch (\Exception $e) {
            Notification::make()
                ->title(__('messages.t_failed_to_submit_reply'))
                ->danger()
                ->send();
        }

        // Reload feedback details to reflect the new reply
        // $this->loadFeedbackDetails();
    }


    /**
     * Renders the FeedbackList view.
     */
    public function render()
    {
        return view('feedback::feedback-list');
    }
}
