<?php

namespace Adfox\Paypal\Http\Controllers\Callback;


use App\Http\Controllers\Controller;
use App\Models\WebhookPackage;
use App\Models\WebhookUpgrade;
use Illuminate\Http\Request;
use App\Settings\PaypalSettings;
use Srmklive\PayPal\Services\PayPal as PayPalClient;
use App\Traits\HandlesPromotionApplication;
use App\Traits\HandlesPackageApplication;
use Illuminate\Support\Str;

class PaypalController extends Controller
{
    use HandlesPackageApplication;
    use HandlesPromotionApplication;

    private $paypalSettings;


    /**
     * Handle the callback from the Stripe payment gateway.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function callback(Request $request)
    {
        $this->paypalSettings = app(PaypalSettings::class);


         // Get transaction id
         $transaction_id = $request->get('order');

         // Check webhook event
         if ($transaction_id) {

            $response = $this->verifyPayment($transaction_id);

            if ( is_array($response) && $response['success'] == TRUE) {

                // Get order id
                $order_id = $response['response']['purchase_units'][0]['payments']['captures'][0]['invoice_id'];

                // Check If Package Management Callback
                if (Str::startsWith($order_id, "PKG-")) {

                        $orderData = WebhookPackage::where('payment_id', $order_id)
                                                ->where('payment_method', 'paypal')
                                                ->where('status', 'pending')
                                                ->firstOrFail();

                        $routeParameters = $this->applyPackages($orderData);

                        return redirect()->route('package-success', $routeParameters);

                } else {
                    $orderData = WebhookUpgrade::where('payment_id', $order_id)
                                            ->where('payment_method', 'paypal')
                                            ->where('status', 'pending')
                                            ->firstOrFail();

                    $routeParameters = $this->applyPromotionsWithRedirect($orderData);

                    return redirect()->route('success-upgrade', $routeParameters);
                }
            }
        }

        return redirect('/');
    }

     /**
     * Verify if payment succeeded
     *
     * @param string $id
     * @return array
     */
    private function verifyPayment($id)
    {
        try {

            // Get payment gateway keys
            $client_id     =$this->paypalSettings->client_id;
            $client_secret = $this->paypalSettings->client_secret;

            // Set gateway config
            $config = [
                'mode' => 'live',
                'live' => [
                    'client_id'     => $client_id,
                    'client_secret' => $client_secret,
                    'app_id'        => '',
                ],
                'sandbox' => [
                    'client_id'     => $client_id,
                    'client_secret' => $client_secret,
                    'app_id'        => '',
                ],
                'payment_action' => 'Sale',
                'currency'       => $this->paypalSettings->currency,
                'notify_url'     => 'https://your-site.com/paypal/notify',
                'locale'         => 'en_US',
                'validate_ssl'   => true,
            ];


            // Set paypal provider and config
            $client = new PayPalClient($config);


            // Set client credentials
            $client->setApiCredentials($config);

            // Get paypal access token
            $client->getAccessToken();

            // Capture this order
            $order  = $client->capturePaymentOrder($id);

            // Check if payment succeeded
            if ( is_array($order) && isset($order['status']) && $order['status'] === 'COMPLETED' ) {

                // Done
                return [
                    'success'  => true,
                    'response' => $order
                ];

            } else {

                // Failed
                return [
                    'success' => false,
                    'message' => __('messages.t_error_payment_failed')
                ];

            }

        } catch (\Throwable $th) {

            // Error
            return [
                'success' => false,
                'message' => __('messages.t_toast_something_went_wrong')
            ];

        }
    }
}
