<?php

namespace Adfox\Packages\Filament\Resources;

use Adfox\Packages\Filament\Resources\PackageResource\Pages;
use App\Models\Package;
use App\Models\Promotion;
use Filament\Forms;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Get;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;

class PackageResource extends Resource
{
    protected static ?string $model = Package::class;

    protected static ?string $navigationGroup = 'Package Management';

    protected static ?string $navigationLabel = 'Packages';

    protected static ?int $navigationSort = 1;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Toggle::make('is_default')
                    ->label('Set as Default')
                    ->helperText('Enable to make this the default package.'),


                Toggle::make('is_enabled')
                    ->label('Activate Package')
                    ->helperText('Turn on to activate this package, off to deactivate it.'),

                TextInput::make('name')
                ->label('Package Name')
                ->required()
                ->placeholder('Enter package name'),



                TextInput::make('duration')
                    ->label('Duration')
                    ->numeric()
                    ->suffix('days')
                    ->required()
                    ->placeholder('Duration in days'),




                Textarea::make('features')
                    ->label('Features')
                    ->required()
                    ->columnSpanFull()
                    ->placeholder('Describe the features of the package')
                    ->rows(3),

                Repeater::make('packageItems')
                    ->label('Ad Count Options')
                    ->relationship()
                    ->schema([
                        TextInput::make('quantity')
                            ->label('Number of Ads')
                            ->numeric()
                            ->required()
                            ->placeholder('e.g., 5'),
                        TextInput::make('price')
                            ->label('Price for this Count')
                            ->numeric()
                            ->required()
                            ->prefix(config('app.currency_symbol'))
                            ->placeholder('Set a price'),
                        Toggle::make('offer_enabled')
                            ->label('Enable Offer')
                            ->live()
                            ->helperText('Activate to set a special offer price.'),
                        TextInput::make('offer_price')
                            ->label('Offer Price')
                            ->required()
                            ->numeric()
                            ->prefix(config('app.currency_symbol'))
                            ->placeholder('Enter offer price')
                            ->visible(fn ($get) => $get('offer_enabled'))
                            ->helperText('Offer price should be less than the regular price.'),
                        DatePicker::make('offer_start')
                            ->label('Offer Start Date')
                            ->required()
                            ->minDate(now())
                            ->native(false)
                            ->visible(fn ($get) => $get('offer_enabled'))
                            ->helperText('Specify the start date for the offer.'),
                        DatePicker::make('offer_end')
                            ->label('Offer End Date')
                            ->required()
                            ->native(false)
                            ->minDate(now())
                            ->visible(fn ($get) => $get('offer_enabled'))
                            ->helperText('Set the end date for the offer.'),

                    ])
                    ->addActionLabel('Add Ad Count Option')
                    ->columnSpanFull()
                    ->columns(2),

                Repeater::make('packagePromotions')
                    ->label('Promotional Options')
                    ->relationship()
                    ->schema([
                        Select::make('promotion_id')
                            ->options(Promotion::all()->pluck('name', 'id'))
                            ->label('Promotion Type')
                            ->helperText('Choose the promotion type to include in the package.')
                            ->required(),
                        Repeater::make('packageItems')
                            ->label('Promotion Usage Limits')
                            ->relationship()
                            ->schema([
                                TextInput::make('quantity')
                                    ->label('Usage Limit')
                                    ->helperText('Specify the number of ads this promotion can be applied to.')
                                    ->numeric()
                                    ->required(),
                                TextInput::make('price')
                                    ->label('Price for Promotion')
                                    ->numeric()
                                    ->required()
                                    ->prefix(config('app.currency_symbol'))
                                    ->placeholder('Enter promotion price'),
                            ])
                            ->addActionLabel('Add Promotion Usage Limit')
                            ->columnSpanFull()
                            ->columns(2),
                    ])
                    ->addActionLabel('Add Promotional Option')
                    ->columnSpanFull()
                    ->columns(2)

            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                ->label('Package Name'),
                TextColumn::make('quantity')
                ->hidden()
                ->label('Ad Count'),
                TextColumn::make('duration')
                ->suffix(' Days')
                ->label('Duration'),
                TextColumn::make('price')
                ->hidden()
                ->label('Price')
                ->prefix(config('app.currency_symbol')),
                ToggleColumn::make('is_enabled')
                ->label('Package Status')
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPackages::route('/'),
            'create' => Pages\CreatePackage::route('/create'),
            'edit' => Pages\EditPackage::route('/{record}/edit'),
        ];
    }
}
