<?php

namespace Adfox\Feedback\Livewire;

use App\Models\Conversation;
use App\Models\Feedback as ModelFeedback;
use App\Models\User;
use Livewire\Component;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Actions\Action;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\ToggleButtons;
use Filament\Notifications\Notification;
use Filament\Support\Enums\MaxWidth;
use Illuminate\Support\Facades\Auth;

class Feedback extends Component implements HasForms, HasActions
{
    use InteractsWithActions;
    use InteractsWithForms;

    public $sellerId;

    public $feedbackDetails;

    public $positiveCount;
    public $neutralCount;
    public $negativeCount;

    public $sellerName;

    public $isMobileHidden = false;

    public ModelFeedback $feedback;

    public function mount($id)
    {
        $this->sellerId = $id;

        $this->verifySellerExists();

        $this->loadSellerName();

        $this->calculateFeedbackCounts();
    }

    protected function loadSellerName()
    {
        $seller = User::find($this->sellerId);
        $this->sellerName = $seller ? $seller->name : __('messages.t_unknown_seller');
    }

     /**
     * Verifies if the seller exists and if the user is not giving feedback to themselves.
     *
     * @return void
     */
    protected function verifySellerExists()
    {
        if (!User::where('id', $this->sellerId)->exists()) {
            abort(404, __('messages.t_seller_not_found'));
        }
    }

    /**
     * Defines the feedback action form.
     *
     * @return Action
     */
    public function feedbackAction(): Action
    {
        return Action::make('feedbackAction')
            ->hidden(fn (): bool => $this->sellerId == Auth::id() || !$this->hasMinConvWithSeller() || $this->hasAlreadyLeftFeedback())
            ->label(__('messages.t_leave_feedback'))
            ->form($this->feedbackFormSchema())
            ->modalWidth(MaxWidth::Large)
            ->modalSubmitActionLabel(__('messages.t_send_feedback'))
            ->action(fn (array $data) => $this->handleFeedbackSubmission($data));
    }

    /**
     * Check if there is at least one conversation with the seller with non-zero messages.
     *
     * @return bool
     */
    protected function hasMinConvWithSeller(): bool
    {
        $buyerId = Auth::id();

        // Check if there exists any conversation between the buyer and the seller with at least one message
        $hasConversation = Conversation::where('buyer_id', $buyerId)
            ->where('seller_id', $this->sellerId)
            ->whereHas('messages')
            ->exists();

        return $hasConversation;
    }

    /**
     * Check if the current user has already left feedback for the seller.
     *
     * @return bool
     */
    protected function hasAlreadyLeftFeedback(): bool
    {
        $buyerId = Auth::id();
        $sellerId = $this->sellerId;

        // Check if feedback already exists from this buyer to the seller
        $existingFeedback = ModelFeedback::where('buyer_id', $buyerId)
                                    ->where('seller_id', $sellerId)
                                    ->exists();

        return $existingFeedback;
    }


   /**
     * Schema for the feedback form.
     *
     * @return array
     */
    protected function feedbackFormSchema(): array
    {
        return [
            ToggleButtons::make('experience')
                ->label(__('messages.t_rate_experience'))
                ->options([
                    'positive' => __('messages.t_positive'),
                    'neutral' => __('messages.t_neutral'),
                    'negative' => __('messages.t_negative'),
                ])
                ->icons([
                    'positive' => 'heroicon-o-face-smile',
                    'neutral' => 'heroicon-o-face-frown',
                    'negative' => 'heroicon-o-face-frown',
                ])
                ->colors([
                    'positive' => 'success',
                    'neutral' => 'warning',
                    'negative' => 'danger',
                ])
                ->inline()
                ->required()
                ->live(),
            Select::make('interaction')
                ->label(__('messages.t_interaction_label'))
                ->options([
                    'successful_purchase' => __('messages.t_successful_purchase'),
                    'deal_failed' => __('messages.t_deal_failed'),
                    'didnt_come_to_deal' => __('messages.t_didnt_come_to_deal'),
                    'cant_reach_seller' => __('messages.t_cant_reach_seller'),
                ])
                ->required()
                ->live(),
            Textarea::make('detail')
                ->label(__('messages.t_detailed_feedback'))
                ->required()
                ->maxLength(1000),
        ];
    }


    /**
     * Handles the submission of feedback.
     *
     * @param  array  $data  Data from the feedback form.
     * @return void
     */
    protected function handleFeedbackSubmission(array $data)
    {
        try {
            ModelFeedback::create([
                'buyer_id' => Auth::id(),
                'seller_id' => $this->sellerId,
                'experience' => $data['experience'],
                'interaction' => $data['interaction'],
                'detail' => $data['detail'],
            ]);

            Notification::make()
                ->title(__('messages.t_feedback_submitted_successfully'))
                ->success()
                ->send();

            $this->dispatch('fetch-feedback');
        } catch (\Exception $e) {
            Notification::make()
                ->title(__('messages.t_failed_to_submit_feedback'))
                ->danger()
                ->send();
        }
    }

    protected function calculateFeedbackCounts()
    {
        $feedbacks = ModelFeedback::where('seller_id', $this->sellerId)->get();

        $this->positiveCount = $feedbacks->where('experience', 'positive')->count();
        $this->neutralCount = $feedbacks->where('experience', 'neutral')->count();
        $this->negativeCount = $feedbacks->where('experience', 'negative')->count();
    }



    /**
     * Render the component view.
     *
     * @return \Illuminate\View\View The view to render.
     */
    public function render()
    {
        return view('feedback::feedback');
    }
}
